/**
*	WARNING: Do Not Remove This Section
*
*	$RCSfile: mcc.c,v $
*	$Revision: 1.31 $
*	$Date: 2006/02/13 14:47:31 $
*	$Author: zapreevis $
*	
*	MRMC is a model checker for discrete-time and continuous-time Markov reward models.
*	It supports reward extensions of PCTL and CSL (PRCTL and CSRL), and allows for the
*	automated verification of properties concerning long-run and instantaneous rewards
*	as well as cumulative rewards.
*	
*	Copyright (C) The University of Twente, 2004-2006.
*	Authors: Maneesh Khattri, Ivan Zapreev
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*	Our contacts:
*		Formal Methods and Tools Group, University of Twente,
*		P.O. Box 217, 7500 AE Enschede, The Netherlands,
*		Phone: +31 53 4893767, Fax: +31 53 4893247,
*		E-mail: mrmc@cs.utwente.nl
*
*	Source description: Contains the main function, starts the execution, does initial vlidation of input parameters.
*/

# include "bitset.h"
# include "sparse.h"
# include "label.h"
# include "runtime.h"
# include "transient.h"
# include "read_rewards.h"
# include "read_tra_file.h"
# include "read_lab_file.h"
# include "read_impulse_rewards.h"
# include <malloc.h>
# include <stdlib.h>
# include <sys/types.h>

#define CSL_MODE_STR "csl"
#define PCTL_MODE_STR "pctl"
#define PRCTL_MODE_STR "prctl"
#define CSRL_MODE_STR "csrl"
#define CSRL_MODE_VMV_TEST_STR "vmv_test"
#define CSRL_MODE_VMV_TEST -100
#define TOPRINT 1 

/**
* This part simply prints the program usage info.
*/
usage()
{
	printf("Usage: mrmc <mode> <.tra file> <.lab file> <.rew file> <.rewi file>\n");
	printf("\t<mode>\t\t- could be one of {%s, %s, %s, %s}.\n",CSL_MODE_STR, PCTL_MODE_STR, PRCTL_MODE_STR, CSRL_MODE_STR);
	printf("\t<.tra file>\t- is the file with the matrix of transitions.\n");
	printf("\t<.lab file>\t- contains labeling.\n");
	printf("\t<.rew file>\t- contains state rewards (for PRCTL/CSRL).\n");
	printf("\t<.rewi file>\t- contains impulse rewards (for CSRL, optional).\n");
	printf("\nNote: In the '.tra' file transitions should be ordered by rows and columns!\n\n");
}

/**
* This part simply prints the programm Intro message
*/
print_intro()
{
	printf(" ---------------------------------------------------------------- \n");
	printf("|                The Markov Reward Model Checker                 |\n");
	printf("|                     MRMC version 1.0 beta                      |\n");
	printf("|        Copyright (C) The University of Twente, 2004-2006.      |\n");
	printf("|                           Authors:                             |\n");
	printf("| Joost-Pieter Katoen, Maneesh Khattri, Ivan Zapreev, Tim Kemna  |\n");
	printf("|         MRMC is distributed under the GPL conditions           |\n");
	printf("|         (GPL states for GNU General Public License)            |\n");
	printf("|         The product comes with ABSOLUTELY NO WARRANTY.         |\n");
	printf("| This is free software, and you are welcome to redistribute it. |\n");
	printf(" ---------------------------------------------------------------- \n");
}

/**
* This method determines the running mode depending on the command line parameters
* It checks whether we will work with CTMC or DTMC
*/
int setRunningMode(int argc, char * argv[], int id)
{
	char * mode = argv[id];
	
	if( strcmp(mode,CSL_MODE_STR) == 0 )
	{
		printf("Running in the CSL model checking mode.\n");
		setRunMode(CSL_MODE);
		return CSL_MODE;
	}
	else if( strcmp(mode,PCTL_MODE_STR) == 0 )
	{
		printf("Running in the PCTL model checking mode.\n");
		setRunMode(PCTL_MODE);
		return PCTL_MODE;
	}
	else if( strcmp(mode,PRCTL_MODE_STR) == 0)
	{
		printf("Running in the PRCTL model checking mode.\n");
		setRunMode(PRCTL_MODE);
		return PRCTL_MODE;
	}
	else if( strcmp(mode,CSRL_MODE_STR) == 0)
	{
		printf("Running in the CSRL model checking mode.\n");
		setRunMode(CSRL_MODE);
		return CSRL_MODE;
	}
	else if ( strcmp(mode,CSRL_MODE_VMV_TEST_STR) == 0 )
	{
		return CSRL_MODE_VMV_TEST;
	}
	else
	{
		printf("ERROR: The '%s' mode is not one of {%s, %s, %s, %s}.\n",mode,CSL_MODE_STR, PCTL_MODE_STR, PRCTL_MODE_STR, CSRL_MODE_STR);
		usage();
		exit(1);
	}
}

/**
* This method simply tests matrix vector and vector matrix multiplication
* @param space the matrix to test
*/
void testVMV(sparse *space)
{
	int i;
	const int N = space->rows;
	double *pV,*pR;
	struct timeval tt1, tt2;
	//Allocate arrays
	pV = (double *)calloc(N, sizeof(double));
	pR = (double *)calloc(N, sizeof(double));
	//Initialize the vector
	for(i=0; i < N; i++)
	{
		pV[i] = 2.0;
	}

	//Do computations M*v
	gettimeofday(&tt1, NULL);
	multiply_mtx_MV( space, pV, pR );
	gettimeofday(&tt2, NULL);
	printf("Time for M*v: %ld micro sec(s)\n",
		(tt2.tv_sec-tt1.tv_sec)*1000000+(tt2.tv_usec-tt1.tv_usec));

	//Do computations v*M
	gettimeofday(&tt1, NULL);
	multiply_mtx_TMV( space, pV, pR );
	gettimeofday(&tt2, NULL);
	printf("Time for v*M: %ld micro sec(s)\n",
		(tt2.tv_sec-tt1.tv_sec)*1000000+(tt2.tv_usec-tt1.tv_usec));
	free(pV);
	free(pR);
	free_sparse_ncolse(space);
}

main(int argc, char *argv[])
{
	print_intro();

	sparse *space=NULL;
	sparse *rewi=NULL;
	labelling *labellin=NULL;
	double * rew=NULL;
	bitset *b=NULL;
	unsigned int sp1=0,sp2=0;
	if(argc >= 2)
	{
		/*Get the running mode*/
		int mode = setRunningMode(argc,argv,1);
		if( mode == CSRL_MODE_VMV_TEST )
		{
			space = read_tra_file(argv[2]);
			testVMV(space);
			return 0;
		}
		#ifndef MAC_OS
			sp1 = mallinfo().uordblks;
		#endif
		if( ((mode==CSL_MODE || mode==PCTL_MODE) && argc!=4) ||
		    (mode==CSRL_MODE && (argc < 5 || argc > 6)) || (mode==PRCTL_MODE && argc!=5) )
		{
			usage();
			exit(1);
		}
		space = read_tra_file(argv[2]);
		if(space==NULL)
		{
			printf("ERROR: The .tra file '%s' was not found.\n",argv[2]);
			exit(1);
		}
		set_state_space(space);
		labellin = read_lab_file(space->rows, argv[3]);
		if(labellin==NULL)
		{
			printf("ERROR: The .lab file '%s' was not found.\n",argv[3]);
			exit(1);
		}
		set_labeller(labellin);
		if ( mode == PRCTL_MODE || mode == CSRL_MODE )
		{
			rew = read_rew_file(space->rows, argv[4]);
			if ( rew == NULL )
			{
				printf("ERROR: The .rew file '%s' was not found.\n",argv[4]);
				exit(1);
			}
			setRewards(rew);
		}
		if ( mode == CSRL_MODE && argc==6 )
		{
			rewi = read_impulse_rewards(argv[5], space->rows);
			if ( rewi == NULL )
			{
				printf("ERROR: The .rewi file '%s' was not found.\n",argv[5]);
				exit(1);
			}
			set_impulse_rewards(rewi);
		}
		#ifndef MAC_OS
			sp2 = mallinfo().uordblks;
		#endif	
		printf("Space Occupied:: %d Bytes.\n", (sp2-sp1));
		printf("Type 'help' to get help.\n>>");
		while( yyparse() )
		{
			b = get_result_bitset();
			if(TOPRINT)
				print_bitset_states(b);
			printf(">>");
		}
		free_sparse_ncolse(space);
		if(rewi!=NULL)
			free_sparse_ncolse(rewi);
		free_labelling(labellin);
		free_rewards();
		free_e();
	}
	else
	{
		usage();
	}
	//printf("*****************************************************************\n");
	//printf("*                The Markov Reward Model Checker                *\n");
	//printf("*                        MRMC v1.0 (beta)                       *\n");
	//printf("*              This product is (was) developed by:              *\n");
	//printf("* Joost-Pieter Katoen, Maneesh Khattri, Ivan Zapreev, Tim Kemna *\n");
	//printf("*             The University of Twente 2004-2006                *\n");
	//printf("*****************************************************************\n");
}
