/**
*	WARNING: Do Not Remove This Section
*
*	$RCSfile: bitset.c,v $
*	$Revision: 1.20 $
*	$Date: 2006/02/13 14:47:32 $
*	$Author: zapreevis $
*	
*	MRMC is a model checker for discrete-time and continuous-time Markov reward models.
*	It supports reward extensions of PCTL and CSL (PRCTL and CSRL), and allows for the
*	automated verification of properties concerning long-run and instantaneous rewards
*	as well as cumulative rewards.
*	
*	Copyright (C) The University of Twente, 2004-2006.
*	Authors: Maneesh Khattri, Ivan Zapreev
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*	Our contacts:
*		Formal Methods and Tools Group, University of Twente,
*		P.O. Box 217, 7500 AE Enschede, The Netherlands,
*		Phone: +31 53 4893767, Fax: +31 53 4893247,
*		E-mail: mrmc@cs.utwente.nl
*
*	Source description: Manage a Bitset - For the satisfaction relation in MCC.
*	Uses: Definition of bitset - bitset.h
*/

# include "macro.h"
#ifdef MAC_OS
	# include <stdlib.h>
#else
	# include <malloc.h>
#endif
# include <math.h>
# include "bitset.h"
# include "macro.h"

/*****************************************************************************
name		: get_new_bitset
role       	: get a new bitset.
@param     	: int n: indicates the size of the required bitset.
@return		: bitset *: returns a pointer to a new bitset.
remark 	        : A bitset is composed of an array of 32-bit unsigned integers.
******************************************************************************/
bitset * get_new_bitset(int n)
{
	int d = n/32+1;
	bitset * newbitset = (bitset *)calloc(1,sizeof(bitset));
	if(n&&d) newbitset->bytesp = (bytes *)calloc(d,sizeof(bitset));
	newbitset->n = n;
	return newbitset;
}

/*****************************************************************************
name		: or
role       	: obtain the disjunction of two bitsets.
@param     	: bitset *a: One of the operands to the disjunction operation.
@param		: bitset *b: The second operand to the disjuction operation.
@return		: bitset *: returns a pointer to the result of disjunction.
remark 	        : The size of the operands must be the same else a NULL pointer is
		    returned as the result.
******************************************************************************/
bitset * or(bitset *a, bitset *b)
{
	int i, n=a->n, d;
	bitset *result = NULL;
	if(n==b->n)
	{
		result = get_new_bitset(n);
		d = n/32+1;
		for(i=0;i<d;i++) result->bytesp[i].bits = a->bytesp[i].bits | b->bytesp[i].bits;
	}
	return result;
}

/*****************************************************************************
name		: or_result
role       	: obtain the disjunction of two bitsets and put result in arg2.
@param     	: bitset *a: One of the operands to the disjunction operation.
@param		: bitset *b: The second operand to the disjunction operation.
******************************************************************************/
void or_result(bitset *a, bitset *result)
{
	int i, n=a->n, d;
	if(n==result->n)
	{
		d = n/32+1;
		for(i=0;i<d;i++) result->bytesp[i].bits = a->bytesp[i].bits | result->bytesp[i].bits;
	}
}

/*****************************************************************************
name		: xor
role       	: obtain the exclusive-or of two bitsets.
@param     	: bitset *a: One of the operands to the exclusive-or operation.
@param		: bitset *b: The second operand to the exclusive-or operation.
@return		: bitset *: returns a pointer to the result of exclusive-or
remark 	        : The size of the operands must be the same else a NULL pointer is
		    returned as the result.
******************************************************************************/
bitset * xor(bitset *a, bitset *b)
{
	int i, n=a->n, d;
	bitset *result = NULL;
	if(n==b->n)
	{
		result = get_new_bitset(n);
		d = n/32+1;
		for(i=0;i<d;i++) result->bytesp[i].bits = a->bytesp[i].bits ^ b->bytesp[i].bits;
	}
	return result;
}

/*****************************************************************************
name		: xor_result
role       	: obtain the xor of two bitsets and put result in arg2.
@param     	: bitset *a: One of the operands to the x-or operation.
@param		: bitset *b: The second operand to the x-or operation.
******************************************************************************/
void xor_result(bitset *a, bitset *result)
{
	int i, n=a->n, d;
	if(n==result->n)
	{
		d = n/32+1;
		for(i=0;i<d;i++) result->bytesp[i].bits = a->bytesp[i].bits ^ result->bytesp[i].bits;
	}
}

/*****************************************************************************
name		: and
role       	: obtain the conjunction of two bitsets.
@param     	: bitset *a: One of the operands to the conjunction operation.
@param		: bitset *b: The second operand to the conjunction operation.
@return		: bitset *: returns a pointer to the result of conjunction.
remark 	        : The size of the operands must be the same else a NULL pointer is
		    returned as the result.
******************************************************************************/
bitset * and(bitset *a, bitset *b)
{
	int i, n=a->n, d;
	bitset *result = NULL;
	if(n==b->n)
	{
		result = get_new_bitset(n);
		d = n/32+1;
		for(i=0;i<d;i++) result->bytesp[i].bits = a->bytesp[i].bits & b->bytesp[i].bits;
	}
	return result;
}

/*****************************************************************************
name		: and_result
role       	: obtain the conjunction of two bitsets and put result in arg2.
@param     	: bitset *a: One of the operands to the conjunction operation.
@param		: bitset *b: The second operand to the conjunction operation.
******************************************************************************/
void and_result(bitset *a, bitset *result)
{
	int i, n=a->n, d;
	if(n==result->n)
	{
		d = n/32+1;
		for(i=0;i<d;i++) result->bytesp[i].bits = a->bytesp[i].bits & result->bytesp[i].bits;
	}
}

/*****************************************************************************
name		: not
role       	: obtain the complement of a bitset.
@param     	: bitset *a: The operand to the complement operation.
@return		: bitset *: returns a pointer to the result of complement.
remark 	        :
******************************************************************************/
bitset * not(bitset *a)
{
	int i, n=a->n, d;
	bitset *result = NULL, *dummy = NULL;
	dummy = get_new_bitset(n); fill_bitset_one(dummy);
	result = xor(a, dummy);
	free_bitset(dummy);
	return result;
}

/*****************************************************************************
name		: not_result
role       	: obtain the conjunction of two bitsets and put result in arg2.
@param     	: bitset *result: One of the operands to the complement operation.
******************************************************************************/
void not_result(bitset *result)
{
	int i, n=result->n, d;
	bitset *dummy = NULL;
	dummy = get_new_bitset(n); fill_bitset_one(dummy);
	xor_result(dummy, result);
	free_bitset(dummy);
}

/*****************************************************************************
name		: get_bit_val
role       	: obtain the value of a specified bit of a bitset.
@param     	: bitset *a: One of the bits of this bitset is of interest.
@param		: int i: The position of the bit of interest in the bitset.
@param		: bit * (result by reference): The bit of interest is returned
		    	in this parameter.
@return		: int: indicates the success/failure of the get operation.
			 -1: invalid i, 1: success
remark 	        :
******************************************************************************/
int get_bit_val(bitset *a, int i, bit *result)
{
	int d, r;
	unsigned long int val;
	if(i<0||i>=a->n) return -1;
	d=i/32; r = (i%32); val=1<<r;
	val = (a->bytesp[d].bits&val)>>r;
	result->b = val;
	return 1;
}

/*****************************************************************************
name		: set_bit_val
role       	: set the value of a specified bit of a bitset.
@param     	: bitset *a: One of the bits of this bitset is to be changed.
@param		: int i: The position of the bit of interest in the bitset.
@param		: bit * val: The new value for the bit of interest.
@return		: int: indicates the success/failure of the get operation.
			 -1: invalid i, 1: success
remark 	        :
******************************************************************************/
void set_bit_val(bitset *a, int i, const bit *val)
{
	int d;
	unsigned long int value=1;

	if(i<0||i>=a->n)
	{
		printf("ERROR: trying to set a value to bit number %d, which is out of bound [0,%d]\n", i , a->n-1);
		exit(1);
	}

	d=i/32;
	value<<=(i%32);
	if(val->b)
		a->bytesp[d].bits|=value;
	else
		a->bytesp[d].bits&=(~value);
}

/*****************************************************************************
name		: print_bitset_states
role       	:print the position of the bits in the given bitset whose value
		 is 1.
@param     	: bitset *a: The bitset to be printed.
@return		: 
remark 	        :
******************************************************************************/
void print_bitset_states(bitset *a)
{
	int i;
	bit getb;
	if( a )
	{
		printf("states = {");
		for(i=0;i<a->n;i++)
		{
			get_bit_val(a,i,&getb);
			if(getb.b)
				printf("%d, ", i+1);
		}
		printf("}\n");
	}else{
		printf("WARNING: Trying to print a bitset for a NULL pointer.\n");
	}
}

/*****************************************************************************
name		: free_bitset
role       	: free the given bitset.
@param     	: bitset *a: The bitset to be freed.
@return		: 
remark 	        :
******************************************************************************/
void free_bitset(bitset *a)
{
	if( a )
	{
		if(a->bytesp) free(a->bytesp);
		free(a);
	}
}

/*****************************************************************************
name		: fill_bitset_one
role       	: fills the given bitset with one.
@param     	: bitset *a: The bitset to be filled.
@return		: bitset *: returns a pointer to the result of the filling
		  operation.
remark 	        :
******************************************************************************/
void fill_bitset_one(bitset *a)
{
	int i, d = a->n/32+1;
	unsigned long int val = pow(2,32)-1;
	for(i=0; i<d; i++) a->bytesp[i].bits = a->bytesp[i].bits | val;
}

/*****************************************************************************
name		: is_bitset_zero
role       	: checks if the bitset contains only zeros.
@param     	: bitset *a: The bitset to be checked.
@return		: 0: bitset contains a non-zero, 1: bitset contains only zeros.
******************************************************************************/
int is_bitset_zero(bitset *a)
{
	int i, d = a->n/32+1, flag=1;
	for(i=0; i<d; i++)
		if(a->bytesp[i].bits!=0)
		{flag=0; break;}
	return flag;
}

/*****************************************************************************
name		: get_idx_next_non_zero
role       	: Get the Index of the next non-zero element.
@param     	: bitset *a: The bitset to be checked.
@param     	: int idx: The Present index.
@return		: next index.
******************************************************************************/
int get_idx_next_non_zero(bitset *a, int idx)
{
	int i, j, d = a->n/32+1, r = (idx%32), p=idx/32, flag=0, r_val;
	unsigned long int val = 1;
	unsigned long int bits;
	for(i=p; i<d; i++)
	{
		bits = a->bytesp[i].bits;
		if(bits!=0)
		{
			if(i==p) {j=r+1;bits=bits>>j;}
			else j=0;
			for(;j<32&&bits!=0;j++)
			{
				if((bits&val)!=0){flag = 1; break;}
				bits=bits>>1;
			}
			if(flag!=0) break;
		}
	}
	if(flag!=0) r_val = ((i*32)+j);
	else r_val  = -1;
	return r_val;
}

/*****************************************************************************
name		: count_non_zero
role       	: count the number of non-zero elements in the given bitset.
@param     	: bitset *a: The bitset to be checked.
@return		: the count.
******************************************************************************/
int count_non_zero(bitset *a)
{
	int i=-1, count=0;
	while((i=get_idx_next_non_zero(a, i))!=-1) ++count;
	return count;
}
