/**
*	WARNING: Do Not Remove This Section
*
*	$RCSfile: bscc.c,v $
*	$Revision: 1.36 $
*	$Date: 2006/02/13 14:47:32 $
*	$Author: zapreevis $
*	
*	MRMC is a model checker for discrete-time and continuous-time Markov reward models.
*	It supports reward extensions of PCTL and CSL (PRCTL and CSRL), and allows for the
*	automated verification of properties concerning long-run and instantaneous rewards
*	as well as cumulative rewards.
*	
*	Copyright (C) The University of Twente, 2004-2006.
*	Authors: Ivan Zapreev
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*	Our contacts:
*		Formal Methods and Tools Group, University of Twente,
*		P.O. Box 217, 7500 AE Enschede, The Netherlands,
*		Phone: +31 53 4893767, Fax: +31 53 4893247,
*		E-mail: mrmc@cs.utwente.nl
*
*	Source description: This file contains functions for the BSCCs detection.
*	   The algorithm is based on the Tarjan's algorithm for searching SCCs.
*/

# include "macro.h"
#ifdef MAC_OS
	# include <stdlib.h>
#else
	# include <malloc.h>
#endif

#include "bscc.h"

/******************THE STACK FUNCTIONS****************************************/

/*The stack representative*/
static int * pStack = NULL;
/*The counter of the number of elements in stack and the stack allocated space counter*/
static int top_stack_idx = -1, alloc_stack_size = 0;

/**
* This function deallocates the stack which is used to store states while searching for BSCCs
*/
static void freeStack()
{
	if(alloc_stack_size > 0)
	{
		free(pStack);
		pStack = NULL;
		top_stack_idx = -1; /* For no elements */
		alloc_stack_size = 0;
	}
}

/**
* Stores the int value on the stack
* @param n the int value to be stored
*/
static void pushStack(int n)
{
	/*If the stack size is equal to the number of stored elements then we
	need to allocat new memory for one more element.*/
	if( alloc_stack_size == (top_stack_idx + 1) )
	{
		pStack = (int *) realloc(pStack,(++alloc_stack_size)*sizeof(int));
	}
	/*Store the element*/
	pStack[++top_stack_idx] = n;
}

/**
* Retrieves the stack elements
* @return -1 if the stack is empry, otherwise the element from the top of the stack
*/
static int popStack()
{
	int val = -1;
	if( top_stack_idx >= 0 )
	{
		val = pStack[top_stack_idx--]; /*Decrease the index of the top element "remove element"*/
	}
	return val;
}

/**
* Returns the upper element of the stack withoput removing it from the stack.
* If the stack is empty returns -1.
*/
static int getTopStack()
{
	int val = -1;
	if( top_stack_idx >= 0 )
	{
		val = pStack[top_stack_idx]; 
		/*Do not decrease the top element's index "leave it"*/
	}
	return val;
}

/******************THE BSCCs SEARCH FUNCTIONS*********************************/

/*This is a holder for the rate matrix of the CTMC*/
static sparse * pStateSpace = NULL;

/*This bitset stores the visited states;*/
static bitset *pVisitedStates = NULL;

/*This bitset stores states that belong to some component
or from which we can reach a component;*/
static bitset *pInComponentStates = NULL;

/*This array stores mapping from node ids to the BSCC ids
The index of the array is the index of node and value in the corresponding
Cell is the id of the BSCC to which the given node belongs
If it is 0 then the BSCC was not found yet or it doesn't belong to any BSCC*/
static short * pBSCCs = NULL;

/*The skip variable is used to exit from the recursion
in the visit(int) function when the search should be stopped.*/
static BOOL bGlobalSkip = FALSE;

/*This variable stores the DFS order*/
static int dfs_order = 1;

/*This variable stores the id of the lately found bscc
(Supports up to 255 bsccs)*/
static short bscc_counter = 0;

/**
* @param pBSCCsHolder the structure to retrieve bscc_counter from
* @return the number of BSCCs found so far.
*/
short getBSCCCounter(TBSCCs * pBSCCsHolder)
{
	return pBSCCsHolder->bscc_counter;
}

/**
* This method returns a mapping array from states into the BSCCs' ids.
* @param pBSCCsHolder the structure to retrieve BSCCs mapping from
* @return The array which stores mapping from node ids into the BSCC ids
* The index of the array is the index of node and value in the corresponding
* Cell is the id of the BSCC to which the given node belongs
* If it is 0 then the BSCC was not found yet or it doesn't belong to any BSCC
*/
short * getStatesBSCCsMapping(TBSCCs * pBSCCsHolder)
{
	return pBSCCsHolder->pBSCCs;
}

/**
* This function simply instantiates and returns the pointer to TBSCCs struct
* @param pStateSpaceTmp a state space to work with
* @return an empty TBSCCs structure
*/
TBSCCs * allocateTBSCCs(sparse * pStateSpaceTmp)
{
  /*Allocate structure*/
  TBSCCs * pBSCCsHolder = (TBSCCs *) calloc(1, sizeof(TBSCCs));
  
  pBSCCsHolder->pStateSpace = pStateSpaceTmp;
  pBSCCsHolder->size = pStateSpaceTmp->rows;
  
  /*Create a visited states set if this is the first search*/
  pBSCCsHolder->pVisitedStates = get_new_bitset(pBSCCsHolder->size);
  pBSCCsHolder->pInComponentStates = get_new_bitset(pBSCCsHolder->size);
  pBSCCsHolder->pBSCCs = (short *) calloc(pBSCCsHolder->size, sizeof(short));
  pBSCCsHolder->dfs_order = 1;
  pBSCCsHolder->bscc_counter = 0;
  
  return pBSCCsHolder;
}

/**
* This method is used to deallocate TBSCCs and all associated data.
* pBSCCsHolder the pointer to the structure to be cleaned
*/
void freeTBSCC(TBSCCs * pBSCCsHolder)
{
    if( pBSCCsHolder )
	{
	  if( pBSCCsHolder->pInComponentStates )
	  {
        free_bitset(pBSCCsHolder->pInComponentStates);
	  }
	  if( pBSCCsHolder->pVisitedStates )
	  {
        free_bitset(pBSCCsHolder->pVisitedStates);
      }
	  if( pBSCCsHolder->pBSCCs )
	  {
        free(pBSCCsHolder->pBSCCs);
	  }

	  free(pBSCCsHolder);
	}
}

/*
* This functions checks whether the given state was already visited or not.
* @param i the state id
* @return true if the state was not yet visited.
*/
static BOOL isNotVisited(int i)
{
	bit bitval;
	get_bit_val(pVisitedStates, i, &bitval);
	return !bitval.b;
}

/**
* Marks the state i as visitedfreeStack()
* @param ippBSCCs the state that was visited
*/
static setVisited(int i)
{
	set_bit_val(pVisitedStates, i, &BIT_ON);
}

/***************THE ROOT ARRAY MANAGEMENT PROCEDURES***************************/

/*This array is used to store the root values of the nodes*/
static int * pRoot = NULL;

/**
* This method initializes the root array of length size
*/
static void initRoot(int size)
{
	pRoot = (int *) calloc(size, sizeof(int));
}

/**
* This method is used to free the memory used by the root array
*/
static void freeRoot()
{
	free(pRoot);
	pRoot = NULL;
}

/**
* Retrieves the root value for the given node
* @param v the node id
* @return the root value
*/
static int getRoot(int v)
{
	return pRoot[v];
}

/**
* Stores the root value for the node v
* @param v the node id
* @param val the root value
*/
static void setRoot(int v, int val)
{
	pRoot[v] = val;
}

/****************THE BSCCS LIST MANAGEMENT*************************************/
/*The temporary storage for the lately found BSCCs' ids*/
static int ** ppNewBSCCs = NULL;

/**
* This metod initializes the ppBSCCs array. It allocates the one
* value and stores the zero value there as it is the initial number
* of BSCCs.
*/
static void initBSCCsList()
{
	ppNewBSCCs = (int ** ) calloc( 1, sizeof( int * ) );
	/*Store the initial amount of BSCCs (it is zero)*/
	ppNewBSCCs[0] = 0;
}

/*
* This method is used to free memory used for storing BSCCs
* @param ppLastBSCCs the set of BSCCs returned by the
*                   int ** getNewBSCCs(bitset *)
*                   function.
*/
void freeBSCCs(int ** ppLastBSCCs)
{
  if(ppLastBSCCs)
  {
	int i, num = (int) ppLastBSCCs[0];
	for( i = 1; i <= num ; i++ )
	{
		free(ppLastBSCCs[i]);
	}
	free(ppLastBSCCs);
  }
}

/**
* Returns the found BSCCs list
* @return the list of BSCCS
*/
static int ** getNewBSCCsList()
{
	return ppNewBSCCs;
}

/**
* This method adds a new BSCC component.
* Increases the bscc_counter value by 1
*/
static void addBSCCToTheList()
{
	int lid = (int) ppNewBSCCs[0] + 1;
	ppNewBSCCs[0] = (int *) lid;
	
	/*Extend the ppBSCCs array length, we say ppNewBSCCs[0]+2 because we
	so far have ppBSCCs[0]+1 elements in it and we need one extra*/
	ppNewBSCCs = (int **) realloc( ppNewBSCCs, ( lid + 1 ) * sizeof( int *) );
	/*Check the numner of bsccs*/
	if( bscc_counter == 255 )
	{
		printf("ERROR:The number of BSCCs exceeds 255, this is not supported.\n");
		exit(1);
	}
	/*Allocate memory to store the id and the number of nodes of the BSCC*/
	ppNewBSCCs[lid] = (int *) calloc( 2 , sizeof( int ) );
	/*Increase the number of found BSCCs*/
	ppNewBSCCs[ lid ][0] = ++bscc_counter;
	ppNewBSCCs[ lid ][1] = 0;
}

/**
* increases the counter of nodes of the current BSCC
*/
static void addBSCCNode()
{
	ppNewBSCCs[ (int) ppNewBSCCs[0] ][1]++;
}

/**
* This function checks if the BSCC consist of 1 node and if yes then
* it stores it's id in the ppNewBSCCs[ppNewBSCCs[0]][2] element
*/
static checkForSingleNode(int w)
{
	int lid = (int) ppNewBSCCs[0];
	if( ppNewBSCCs[ lid ][1] == 1 )
	{
		/*Add new element*/
		ppNewBSCCs[ lid ]=(int *)realloc(ppNewBSCCs[lid],3*sizeof(int));
		/*Store value*/
		ppNewBSCCs[lid][2] = w;
	}
}

/**************THE BSCC SEARCH PROCEDURES**************************************/

/**
* This method indicates if the w belongs to the component or not and if
* there exists a path from w to some component or not.
* @param w the node to be tested
* @return TRUE if w is in a component or there is a path from w to some component.
*/
static BOOL isInComponent(int w)
{
	bit bitval;
	get_bit_val(pInComponentStates, w, &bitval);
	return bitval.b;
}

/**
* This procedure sets the corresponding value to the v element of the 
	*pInComponentStates
* bit set. This is used to mark the component as belonging to some state.
*/
static void setInComponent(int v, const bit *pBit)
{
	set_bit_val(pInComponentStates, v, pBit);
}

/**
* This method returns the next after *idx element from the row w. The idx value is then updated.
* @param w the row number
* @param num the number of previously retrieved non zerro element from a row
* @return the value or -1 if all elements have beed processed
*/
static int getNextInARow(int w , int * num)
{
	int result = -1;
	int idx = *num;
	BOOL isDiag = pStateSpace->val[w].diag != 0;
	const total = pStateSpace->ncols[w] + (isDiag? 1:0);
	
	/*If there are some unretrieved elements*/
	if ( total > idx )
	{
		/*If there is a diagonal element treat it first 
		just because it is stored differently*/
		if ( ( idx == 0 ) && isDiag )
		{
			/*There is a non zero diagonal element*/
			result = w;
		}
		else
		{
			/*Retrieve the rest non zero elements*/
			result = pStateSpace->val[w].col[(isDiag?idx-1:idx)];
		}
		*num = *num + 1;
	}
	
	return result;
}

/**
* This metjhod retrieves a BSCC from the stack.
* @param v the root node of the BSCC
*/
static void getBSCCFromStack(int v)
{
	int w;
	addBSCCToTheList();
	do
	{
		w = popStack();
		setInComponent(w, &BIT_ON);
		pBSCCs[w] = bscc_counter;
		addBSCCNode();
	}while(w != v);
	
	checkForSingleNode(w);
}

/**
* This recursive procedure is used to pass through the successive nodes
* in order to detect all the Bottom Strongly Connected Components
* containing the given node. If there is a path from i to some other
* component (SCC) then the procedure exits.
* @param i the initial node
*/
static void visit(int v)
{
	bit unit_bitval;
	int w, num=0, initial_root = dfs_order++;
	
	setRoot(v,initial_root);
	setInComponent(v, &BIT_OFF);
	setVisited(v);
	pushStack(v);

	/*Iterate through all the successive nodes*/
	while( ( w = getNextInARow(v, &num) ) != -1 )
	{
		if( isNotVisited(w) )
		{
			/*Start recursion*/
			visit(w);
			/*Check if we need to exit search*/
			if( bGlobalSkip ) break;
		}
		if( ! isInComponent(w))
		{
			setRoot(v, MIN(getRoot(v), getRoot(w)));
		}
		else
		{
			/*There is a way from v to some component to which w belongs*/
			/*So v can not be a part of BSCC, thus skip.*/
			bGlobalSkip = TRUE;
			/*setInComponent(v, &BIT_ON);*/
			break;
		}
	}
	/*If we did not meet any component yet then it means that there
	can be a BSCC in the stack*/
	if( ! bGlobalSkip )
	{
		if ( getRoot(v) == initial_root )
		{
			/*Found a BSCC let's get it from the stack.*/
			getBSCCFromStack(v);
		}
	}
}

/**
* This function returns the list of ids for a newly found BSCCs
* @param pBSCCsHolder the work structure to store all required data for
*               BSCCs search for a particular state space
* @param pStates the set of states
* @return the two dimensional array (result[][])
* result[1][0] the id of a newly found bsccs,
* result[1][1] the number of elements in a bscc
* result[1][2] the id of the node if result[1][1] = 1
*/
int ** getNewBSCCs(TBSCCs * pBSCCsHolder, bitset *pStates)
{
	const int size = pBSCCsHolder->size;
	int i, elem;
	bit bitval;
	
	/*Init the BSCCs list*/
	initBSCCsList();
	
	/*Allocate the root array;*/
	initRoot(size);
	
	/*Copy data from TBSCC struct to internal variables*/
	pStateSpace = pBSCCsHolder->pStateSpace;
	pVisitedStates = pBSCCsHolder->pVisitedStates;
	pInComponentStates = pBSCCsHolder->pInComponentStates;
	pBSCCs = pBSCCsHolder->pBSCCs;
	bscc_counter = pBSCCsHolder->bscc_counter;
	dfs_order = pBSCCsHolder->dfs_order;
	
	/*Pass througn all the states from the *pStates.*/
	for(i=0; i<size; i++)
	{
		get_bit_val(pStates, i, &bitval);
		if( bitval.b )
		{
			if( isNotVisited(i) )
			{
				bGlobalSkip = FALSE;
				/*Start the recursion*/
				visit(i);
				/*Mark all the components in stack as belonged
				to some component*/
				while( ( elem = popStack() ) != -1 )
					setInComponent(elem, &BIT_ON);
			}
		}
	}
	
	/*Copy stack variable values back to TBSCCs*/
	pBSCCsHolder->bscc_counter = bscc_counter;
	pBSCCsHolder->dfs_order = dfs_order;

	/*Free the stack memory*/
	freeStack();
	freeRoot();
	
	return getNewBSCCsList();
}

