/**
*	WARNING: Do Not Remove This Section
*
*	$RCSfile: sparse.c,v $
*	$Revision: 1.63 $
*	$Date: 2006/02/13 14:47:33 $
*	$Author: zapreevis $
*	
*	MRMC is a model checker for discrete-time and continuous-time Markov reward models.
*	It supports reward extensions of PCTL and CSL (PRCTL and CSRL), and allows for the
*	automated verification of properties concerning long-run and instantaneous rewards
*	as well as cumulative rewards.
*	
*	Copyright (C) The University of Twente, 2004-2006.
*	Authors: Maneesh Khattri, Ivan Zapreev, Tim Kemna
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*	Our contacts:
*		Formal Methods and Tools Group, University of Twente,
*		P.O. Box 217, 7500 AE Enschede, The Netherlands,
*		Phone: +31 53 4893767, Fax: +31 53 4893247,
*		E-mail: mrmc@cs.utwente.nl
*
*	Source description: Manage Sparse Matrix.
*	Uses: DEF: sparse.h
*	Definition of sparse matrix - sparse.h
*/

#include "macro.h"
#ifdef MAC_OS
	#include <stdlib.h>
	#include <strings.h>
#else
	#include <malloc.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include "bitset.h"
#include "sparse.h"

static sparse *s = NULL;
/*****************************************************************************
name		: get_sparse
role       	: get a reference to the sparse matrix.
@return		: sparse *: pointer to the sparse matrix.
remark 	        :
******************************************************************************/
sparse * get_sparse()
{
	return s;
}

/*****************************************************************************
name		: set_sparse
role       	: set a reference to the sparse matrix.
@param     	: sparse * s_in
@return		:
remark 	        : This method has to be called before any other operations on
		  the sparse matrix.
******************************************************************************/
void set_sparse(sparse *s_in)
{
	s=s_in;
}

/*****************************************************************************
name		: free_sparse
role       	: frees the sparse matrix.
@param		: sparse * pM the matrix to remove
remark 	        :
******************************************************************************/
void free_mtx_sparse(sparse * pM)
{
	int i,col,j, rows = pM->rows;
	if(pM)
	{
		for(i=0;i<rows;i++)
		{
			if(pM->ncols[i])
			{
				if(pM->rows==pM->cols)
					free(pM->val[i].back_set);
				free(pM->val[i].col);
				free(pM->val[i].val);
			}
		}
		free(pM->val);
		free(pM->ncols);
		if ( pM->rows == pM->cols && pM->pcols ) free(pM->pcols);
		free(pM);
	}
}

/*****************************************************************************
name		: free_mtx_wrap
role       	: frees the sparse matrix. Does not clean rows.
@param		: sparse * pM the matrix to remove
******************************************************************************/
extern void free_mtx_wrap(sparse *pM)
{
	if(pM)
	{
		free(pM->val);
		free(pM->ncols);
		if ( pM->rows==pM->cols&&pM->pcols ) free(pM->pcols);
		free(pM);
	}
}

/*****************************************************************************
name		: free_sparse
role       	: frees the sparse matrix.
@param		: sparse * pM the matrix to remove
remark 	        :
******************************************************************************/
void free_sparse_ncolse(sparse * pM)
{
	int i,col,j, rows = pM->rows;
	if(pM)
	{
		if(pM->rows==pM->cols)
		{
			for(i=0;i<pM->rows;i++)
				free(pM->val[i].back_set);
		}
		if(pM->val && pM->val[0].col && pM->val[0].val)
			{free(pM->val[0].col); free(pM->val[0].val);}
		free(pM->val);
		free(pM->ncols);
		if ( pM->rows==pM->cols&&pM->pcols ) free(pM->pcols);
		free(pM);
	}
}


/*****************************************************************************
name		: free_sparse
role       	: frees the sparse matrix.
remark 	        :
******************************************************************************/
void free_sparse()
{
	free_mtx_sparse(s);
	s=NULL;
}

/*****************************************************************************
name		: init_sparse
role       	: initialise a matrix
@param		: int rows: number of rows in the matrix.
@param		: int cols: number of cols in the matrix.
@return		: the created sparse matrix
remark 	        : initialises a matrix for the given number of rows and columns
******************************************************************************/
sparse * init_matrix(int rows, int cols)
{
	int i;
	s = (sparse *) calloc (1, sizeof(sparse));
	s->rows = rows;
	s->cols = cols;
	s->val = (values *) calloc (s->rows,(sizeof(values)));
	s->ncols = (int *) calloc (s->rows,(sizeof(int)));
	s->pcols = (int *) calloc (s->rows,(sizeof(int))); /* NOTE:: Works only for square matrix */
	return s;
}

/*****************************************************************************
name		: init_sparse_ncolse		{

role       	: initialise a matrix
@param		: int rows: number of rows in the matrix.
@param		: int cols: number of cols in the matrix.
@return		: the created sparse matrix

remark 	        : initialises a matrix for the given number of rows and columns
******************************************************************************/
sparse * init_matrix_ncolse(int rows, int cols, int *ncolse)
{
	int i, sum=0;
	int *col_val;
	double *val;
	s = (sparse *) calloc (1, sizeof(sparse));
	s->rows = rows;

	s->cols = cols;
	s->val = (values *) calloc (s->rows,(sizeof(values)));
	s->ncols = (int *) calloc (s->rows,(sizeof(int)));
	s->pcols = (int *) calloc (s->rows,(sizeof(int)));/* NOTE:: Works only for square matrix */
	
	for(i=0;i<rows;i++) sum+=ncolse[i];

	col_val = (int *)calloc(sum, sizeof(int));
	val = (double *)calloc(sum, sizeof(double));
	if(col_val==NULL || val==NULL) printf("couldn't allocate space\n");
	sum=0;
	for(i=0;i<rows;i++)
	{
		s->val[i].col = &col_val[sum];
	 	s->val[i].val = &val[sum];
		sum+=ncolse[i];
	}
	return s;
}


/*****************************************************************************
name		: set_mtx_val
role       	: set a value in the matrix.
@param		: sparse * pM the matrix to add element to
@param     	: int row: The row number in which the value is to be set.
@param		: int col: The column number in which the value is to be set.
@param		: double val: The value to be set.
@return		: int: 0: fail, 1: success
remark 	        : return 0: indicates that row/col does not meet the bounds
		  of matrix
******************************************************************************/
unsigned int set_mtx_val(sparse * pM, int row, int col, double val)
{
	int idx = pM->ncols[row], idxp=0;
	bit b; b.b = 1;
	if( !pM || row < 0 || row > pM->rows ||col < 0 || col > pM->cols ) return 0;
	if(val != 0 && row!=col)
	{
		pM->val[row].col = (int *)realloc(pM->val[row].col,(idx+1)*sizeof(int));
		pM->val[row].val = (double *)realloc(pM->val[row].val, (idx+1)*sizeof(double));
		pM->val[row].col[idx] = col;
		pM->val[row].val[idx] = val;
		if(pM->rows==pM->cols)
		{
			idxp=pM->pcols[col];
			pM->val[col].back_set = (int *)realloc(pM->val[col].back_set, (idxp+1)*sizeof(double));
			pM->val[col].back_set[idxp]=row;
			++pM->pcols[col];
		}
		++pM->ncols[row];
	}
	else if(val != 0 && row==col)
		pM->val[row].diag = val;
	return 1;
}

/*****************************************************************************
name		: set_val
role       	: set a value in the matrix.
@param     	: int row: The row number in which the value is to be set.
@param		: int col: The column number in which the value is to be set.
@param		: double val: The value to be set.
@return		: int: 0: fail, 1: success
remark 	        : return 0: indicates that row/col does not meet the bounds
		  of matrix
******************************************************************************/
unsigned int set_val(int row, int col, double val)
{
	return set_mtx_val(s,row,col,val);
}

/*****************************************************************************
name		: set_val_ncolse
role       	: set a value in the matrix.
@param     	: int row: The row number in which the value is to be set.
@param		: int col: The column number in which the value is to be set.
@param		: double val: The value to be set.
@return		: int: 0: fail, 1: success
remark 	        : return 0: indicates that row/col does not meet the bounds
		  of matrix
******************************************************************************/
unsigned int set_val_ncolse(int row, int col, double val)
{
	//Here s->ncols[row] is initially 0.
	int idx = s->ncols[row], idxp=0;
	bit b; b.b=1;
	if(!s || row<0||row>s->rows||col<0||col>s->cols) return 0;
	if(val != 0 && row!=col)
	{
		s->val[row].col[idx] = col;
		s->val[row].val[idx] = val;
		if(s->rows==s->cols)
		{
			idxp=s->pcols[col];
			s->val[col].back_set = (int *)realloc(s->val[col].back_set, (idxp+1)*sizeof(double));
			s->val[col].back_set[idxp]=row;	
			++s->pcols[col];
		}
		++s->ncols[row];
	}
	else if(val != 0 && row==col)
		s->val[row].diag = val;
	return 1;
}

/*****************************************************************************
name		: add_mtx_val
role       	: add a value to an existing one in the matrix.
@param		: sparse * pM the matrix
@param     	: int row: The row number in which the value is to be added.
@param		: int col: The column number in which the value is to be added.
@param		: double val: to be added.
remark 	        :
******************************************************************************/
void add_mtx_val(sparse * pM, int row, int col, double val)
{
	int ncols = pM->ncols[row], i;
	unsigned int found=0;
	if( !pM || row < 0 || row > pM->rows || col <0 || col > pM->cols )
		return;
	if ( row != col )
	{
		for( i=0; i < ncols; i++ )
		{
			if( pM->val[row].col[i] == col )
			{
				pM->val[row].val[i]+=val;
				found=1;
				break;
			}
		}
	}
	else
	{
		pM->val[row].diag+=val;
		found = 1;
	}
	if(found==0)
		set_mtx_val(pM, row, col, val);
}



/*****************************************************************************
name		: get_mtx_next_val
role       	: get the next value in the row of a matrix.
@param		: sparse * pM the matrix to get element from
@param     	: int row: The row number in which the value is to be set.
@param		: int * col (by reference): The present col. (Initially points to NULL)
@param		: double * val (by reference): The present value. (Initially points to NULL)
@param		: int *count (by reference): The count.
@return		: int: 0: fail, 1: success
remark 	        : return 0: value was not found / row/col does not meet bounds.
		  **Does not get the values in the diagonal - Use get_mtx_diag_val**
******************************************************************************/
unsigned int get_mtx_next_val(sparse * pM, int row, int *col, double *val, int *count)
{
	int ncols = pM->ncols[row];
	unsigned int found=0;
	if( !pM || row < 0 || row > pM->rows ) found = 0;
	else if( *count==0 )
	{
		if( 1>ncols ) found=0;
		else{ *count=1;*col=pM->val[row].col[*count-1];*val=pM->val[row].val[*count-1];found=1; }
	}
	else
	{
		if( (*count+1)>ncols ) found=0;
		else{ ++*count;*col=pM->val[row].col[*count-1];*val=pM->val[row].val[*count-1];found=1; }
	}
	return found;
}

/*****************************************************************************
name		: get_mtx_diag_val
role       	: get the diag value in the row of a matrix.
@param		: sparse * pM the matrix to get element from
@param     	: int row: The row number in which the value is to be set.
@param		: double * val (by reference): The value.
@return		: int: 0: fail, 1: success
remark 	        : return 0: value was not found / row/col does not meet bounds.
		  **Does not get the values in the diagonal**
******************************************************************************/
unsigned int get_mtx_diag_val(sparse * pM, int row, double *val)
{
	unsigned int found=0;
	if( pM->val[row].diag!=0 )
	{
		*val = pM->val[row].diag;
		found = 1;
	}
	return found;
}

/*****************************************************************************
name		: get_mtx_val
role       	: get a value in the matrix.
@param		: sparse * pM the matrix to get element from
@param     	: int row: The row number in which the value is to be set.
@param		: int col: The column number in which the value is to be set.
@param		: double * val (by reference): The value.
@return		: int: 0: fail, 1: success
remark 	        : return 0: value was not found / row/col does not meet bounds.
******************************************************************************/
unsigned int get_mtx_val(sparse * pM, int row, int col, double *val)
{
	int ncols = pM->ncols[row], i;
	unsigned int found=0;
	if( !pM || row < 0 || row > pM->rows || col <0 || col > pM->cols )
	{
		return 0;
	}
	if ( row != col )
	{
		for( i=0; i < ncols; i++ )
		{
			if( pM->val[row].col[i] == col )
			{
				*val = pM->val[row].val[i];
				found=1;
				break;
			}
		}
	}
	else if( pM->val[row].diag )
	{
		*val = pM->val[row].diag;
		found = 1;
	}
	return found;
}

/*****************************************************************************
name		: get_val
role       	: get a value in the matrix.
@param     	: int row: The row number in which the value is to be set.
@param		: int col: The column number in which the value is to be set.
@param		: double * val (by reference): The value.
@return		: int: 0: fail, 1: success
remark 	        : return 0: value was not found / row/col does not meet bounds.
		  SLOW
******************************************************************************/
unsigned int get_val(int row, int col, double *val)
{
	return get_mtx_val(s, row, col, val);
}

/*****************************************************************************
name		: get_row_sums
role       	: add the elements in each row.
@param		: sparse * pM the matrix to work with
@return		: double *: row sums for each row.
remark 	        :
******************************************************************************/
double * get_mtx_row_sums(sparse * pM)
{
	int i,j, size = pM->rows;
	double *row_sum = (double *)calloc(size, sizeof(double));
	for( i=0; i < size; i++ )
	{
		int col = pM->ncols[i];
		double result=0.0;
		for( j=0; j < col; j++ )
		{
			result += pM->val[i].val[j];
		}
		row_sum[i] = result + pM->val[i].diag;
	}
	return row_sum;
}

/*****************************************************************************
name		: get_row_sums
role       	: add the elements in each row.
@return		: double *: row sums for each row.
remark 	        :
******************************************************************************/
double * get_row_sums()
{
	return get_mtx_row_sums(s);
}

/*****************************************************************************
name		: get_mtx_cer_row_sums
role       	: add elements in certain rows and return the vector of values.
@param		: sparse * pM the matrix to work with
@param		: int * pIds the ids of the rows for which the sums should be computed
			the pIds[0] contains the number of row ids
@return		: the rows' sums
******************************************************************************/
double * get_mtx_cer_row_sums(sparse * pM, int * pIds)
{
	int i, j;
	int * pNCols = pM->ncols;
	values * pRows = pM->val;
	int num = pIds[0];
	
	double * pE = (double *)calloc(num, sizeof(double));
	for( i=1; i<= num; i++ )
	{
		int col = pNCols[ pIds[i] ];
		values row = pRows[ pIds[i] ];
		
		/*sum all the non diagonal elements first*/
		double result=0.0;
		for( j=0; j < col; j++ )		
			result += row.val[j];
		/*Add diagonal value*/
		pE[ i-1 ] = result + row.diag;
	}
	return pE;
}

/*****************************************************************************
name		: mult_mtx_const
role       	: multiply all elements in the matrix with a constant.
@param		: sparse * pM the matrix to multiply
@param     	: double constant: The constant value.
******************************************************************************/
void mult_mtx_const(sparse * pM, double constant)
{
	int i,j;
	int length = pM->rows;
	int * pNCols = pM->ncols;
	values * pRows = pM->val;
	values * pRow = NULL;
	double * pVal;
	
	for( i=0; i < length; i++ )
	{
		int col = pNCols[i];
		pRow = & pRows[i];
		pVal = pRow->val;
		for( j=0; j < col; j++ )		
			pVal[j] *= constant;
		pRow->diag *= constant;
	}
}

/*****************************************************************************
name		: mult_const
role       	: multiply all elements in the matrix with a constant.
@param     	: double constant: The constant value.
******************************************************************************/
void mult_const(double constant)
{
	mult_mtx_const(s, constant);
}

/*****************************************************************************
name		: mult_mtx_cer_const
role       	: multiply all elements in the matrix with a constant.
@param		: sparse * pM the matrix to multiply
@param     	: double constant: The constant value.
@param		: int * pValidStates the valid rows to multiply pValidStates[0]
		  is the number of states (rows) in the array
******************************************************************************/
void mult_mtx_cer_const(sparse * pM, double constant, int * pValidStates)
{
	int i, j, id;
	int length = pValidStates[0];
	int * pNCols = pM->ncols, col;
	values * pRows = pM->val;
	values * pRow = NULL;
	double * pVal;
	
	for( i=1; i <= length; i++ )
	{
		id = pValidStates[i];
		col = pNCols[ id ];
		pRow = & pRows[ id ];
		pVal = pRow->val;
		for( j=0; j < col; j++ )		
			pVal[j] *= constant;
		pRow->diag *= constant;
	}
}

/*****************************************************************************
name		: mult_mtx_cer_const_array
role       	: multiply all elements in the matrix with a constant.
@param		: sparse * pM the matrix to multiply
@param     	: double * constant: The constant value.
@param		: int * pValidStates the valid rows to multiply pValidStates[0]
		  is the number of states (rows) in the array
******************************************************************************/
void mult_mtx_cer_const_array(sparse * pM, double *constant, int * pValidStates, int invert)
{
	int i, j, id, col;
	int length = pValidStates[0];
	double * pVal;
	double cons;
	
	for( i=1; i <= length; i++ )
	{
		id = pValidStates[i];
		col = pM->ncols[ id ];
		pVal = pM->val[id].val;
		cons=constant[id];
		if (invert&&cons) cons=1/cons;
		for( j=0; j < col; j++ )
			pVal[j] *= cons;
		pM->val[id].diag *= cons;
	}
}

/*****************************************************************************
name		: add_diagonal
role       	: add the elements of the input array to the diagonal one-one.
@param     	: double *diagonal: The addendum to the diagonal.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
		  size should be correct.
******************************************************************************/
unsigned int add_diagonal(double *diagonal)
{
	int i, size;
	if(!s) return 0;
	size = s->rows;
	for( i=0; i < size; i++ )
		s->val[i].diag += diagonal[i];
	return 1;
}

/*****************************************************************************
name		: add_mtx_diagona
role       	: add the elements of the input array to the diagonal one-one.
@param		: sparse * pM the matrix to work with
@param     	: double *diagonal: The addendum to the diagonal.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
		  size should be correct.
******************************************************************************/
void add_mtx_diagonal(sparse * pM, double *diagonal)
{
	int i, size;
	values *pRows = pM->val;
	size = pM->rows;
	for( i=0; i < size; i++ )
		pRows[i].diag += diagonal[i];
}

/*****************************************************************************
name		: sub_diagonal
role       	: sub the elements of the input array to the diagonal one-one.
@param		: sparse * pM the matrix to work with
@param     	: double *diagonal: The addendum to the diagonal.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
		  size should be correct.
******************************************************************************/
unsigned int sub_mtx_diagonal(sparse * pM, double *diagonal)
{
	int i, size;
	if(!pM) return 0;
	size = pM->rows;
	for( i=0; i < pM->rows; i++ )
	{
		pM->val[i].diag -= diagonal[i];
	}
	return 1;
}

/*****************************************************************************
name		: sub_diagonal
role       	: sub the elements of the input array to the diagonal one-one.
@param     	: double *diagonal: The addendum to the diagonal.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
		  size should be correct.
******************************************************************************/
unsigned int sub_diagonal(double *diagonal)
{
	return sub_mtx_diagonal(s, diagonal);
}

/****************************************************************************
name		: add_mtx_cer_diagonal
role		: diag elements = cons - diag elements
@param		: sparse * pM the matrix to work with
@param		: int * pIds the ids of the rows in which diagonal elements
		  should be reduced by the correcponding pE values
		  the pIds[0] contains the number of row ids
@param		: int constant: an operand (cons)
******************************************************************************/
void add_cer_cons_diagonal_mtx(sparse *pM, int * pIds, int constant)
{
	int i;
	values * pRows = pM->val;
	int num = pIds[0];
	for( i=1; i <= num; i++ )
		pRows[ pIds[i] ].diag = constant+pRows[ pIds[i] ].diag;
}

/****************************************************************************
name		: sub_cer_cons_diagonal_mtx
role		: diag elements = cons - diag elements
@param		: sparse * pM the matrix to work with
@param		: int * pIds the ids of the rows in which diagonal elements
		  should be reduced by the correcponding pE values
		  the pIds[0] contains the number of row ids
@param		: int constant: an operand (cons)
******************************************************************************/
void sub_cer_cons_diagonal_mtx(sparse *pM, int * pIds, int constant)
{
	int i;
	values * pRows = pM->val;
	int num = pIds[0];
	for( i=1; i <= num; i++ )
	{
		pRows[ pIds[i] ].diag = constant-pRows[ pIds[i] ].diag;
	}
}

/****************************************************************************
name		: sub_mtx_cer_diagonal
role		: Subtract elements of the pE array from the predefined diagonal
		  elements of the matrix
@param		: sparse * pM the matrix to work with
@param		: int * pIds the ids of the rows in which diagonal elements
		  should be reduced by the correcponding pE values
		  the pIds[0] contains the number of row ids
@param		: double * pE: The elements to be subtracted
******************************************************************************/
void sub_mtx_cer_diagonal(sparse *pM, int * pIds, double * pE)
{
	int i;
	values * pRows = pM->val;
	int num = pIds[0];
	for( i=1; i <= num; i++ )
	{
		pRows[ pIds[i] ].diag -= pE[i-1];
	}
}

/*****************************************************************************
name		: add_mtx_cons_diagonal
role       	: add a constant to the all the elements of the diagonal.
@param		: sparse * pM the matrix to work with
@param     	: double constant: The constant value.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
******************************************************************************/
unsigned int add_mtx_cons_diagonal(sparse * pM, double cons)
{
	int i, size;
	if(!pM) return 0;
	size = pM->rows;
	for(i=0;i<pM->rows;i++)
		pM->val[i].diag += cons;
	return 1;
}

/*****************************************************************************
name		: add_cons_diagonal
role       	: add a constant to the all the elements of the diagonal.
@param     	: double constant: The constant value.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
******************************************************************************/
unsigned int add_cons_diagonal(double cons)
{
	return add_mtx_cons_diagonal(s, cons);
}

/*****************************************************************************
name		: sub_cons_diagonal
role       	: sub a all the elements of the diagonal from a constant.
@param     	: double constant: The constant value.
@param     	: int flag: 0: cons-diag_i, flag: 1 diag_i-cons
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
******************************************************************************/
unsigned int sub_cons_diagonal(double cons, int flag)
{
	int i, size;
	if(!s) return 0;
	size = s->rows;
	for(i=0;i<s->rows;i++)
		if(flag)
			s->val[i].diag = s->val[i].diag-cons;
		else
			s->val[i].diag = cons-s->val[i].diag;
	return 1;
}

/*****************************************************************************
name		: set_mtx_row
role       	: set a row in the matrix.
		  NOTE: This function copies the values structure that is why change
		  of fields in the original row lateron will not affect the one which
		  was set by this function. For instance change of the diagonal element
		  in the original matrix will not affect the row added to the other matrix
		  by this method.
@param		: sparse * pM the matrix to work with
@param     	: int row: The row number in which the value is to be set.
@param		: int ncol: The number of valid columns (non-zero) in the given row.
@param		: values rowval: The row value to be set.
@return		: int: 0: fail, 1: success
remark 	        : return 0: indicates that row/col does not meet the bounds
		  of matrix
******************************************************************************/
unsigned int set_mtx_row(sparse * pM, int row, int ncols, values rowval)
{
	if( !pM || row < 0 || row > pM->rows )
		return 0;
	pM->ncols[row] = ncols;
	pM->val[row] = rowval;
	return 1;
}

/*****************************************************************************
name		: set_row
role       	: set a row in the matrix.
		  NOTE: This function copies the values structure that is why change
		  of fields in the original row lateron will not affect the one which
		  was set by this function. For instance change of the diagonal element
		  in the original matrix will not affect the row added to the other matrix
		  by this method.
@param     	: int row: The row number in which the value is to be set.
@param		: int ncol: The number of valid columns (non-zero) in the given row.
@param		: values rowval: The row value to be set.
@return		: int: 0: fail, 1: success
remark 	        : return 0: indicates that row/col does not meet the bounds
		  of matrix
******************************************************************************/
unsigned int set_row(int row, int ncols, values rowval)
{
	return set_mtx_row(s, row, ncols, rowval);
}

/*****************************************************************************
name		: print_sparse
role       	: print the matrix.
@param		: sparse * pM the matrix to work with
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
******************************************************************************/
unsigned int print_mtx_sparse(sparse * pM)
{
	int i,j;
	if( !pM ) return 0;
	for( i=0; i < pM->rows; i++ )
	{
		int col = pM->ncols[i];
		printf("ncols[%d]:: %d\n", i, col);
		for( j=0; j < col; j++ )
			printf("[%d][%d]=%lf\n",i,(int)pM->val[i].col[j],pM->val[i].val[j]);
	}
	for( i=0; i < pM->rows; i++)
		printf("diag:: [%d][%d]=%lf\n", i, i, pM->val[i].diag);
	if(pM->rows==pM->cols)
	{
		printf("********************* Printing Back Sets *************************************\n");
		for( i=0; i < pM->rows; i++)
		{
			printf("state: %d : BS : [ ", i+1);
			for(j=0;j<pM->pcols[i];j++)
				printf("%d, ", pM->val[i].back_set[j]+1);
			printf("]\n");
		}
		printf("******************************************************************************\n");
	}
	return 1;
}

/*****************************************************************************
name		: print_sparse
role       	: print the matrix.
@return		: int: 0: fail, 1: success
remark 	        : return 0: sparse matrix is not valid.
******************************************************************************/
unsigned int print_sparse()
{
	return print_mtx_sparse(s);
}

/*****************************************************************************
name		: multiply_mtx_MV
role       	: multiply a matrix with a vector.
@param		: sparse * pM: operand matrix.
@param     	: double *vec: The operand vector.
@param		: double *res: The resulting vector.
@return		: int: 0: fail, 1: success
remark 	        : size should be correct.
******************************************************************************/
void multiply_mtx_MV(sparse * pM, double *vec, double *res)
{
	int i, j, rows = pM->rows, cols, *col, *ncols=pM->ncols;
	double *val, result;
	for(i=0;i<rows;i++)
	{
		cols = *ncols++;
		result = pM->val[i].diag*vec[i];
		if(cols!=0)
		{
			col = pM->val[i].col; val = pM->val[i].val;
			for(j=0;j<cols;j++)
				result+=vec[*col++]*(*val++);
		}
		res[i]=result;
	}
}


/*****************************************************************************
name		: multiplyMV
role       	: multiply a matrix with a vector.
@param     	: double *vec: The operand vector.
@param		: double *res: The resulting vector.
@return		: int: 0: fail, 1: success
remark 	        : size should be correct.
******************************************************************************/
unsigned int multiplyMV(double *vec, double *res)
{
	multiply_mtx_MV(s, vec, res);
	return 1;
}

/*****************************************************************************
name		: multiply_mtx_cer_MV
role       	: multiply certain rows of a matrix with a vector.
@param		: sparse * pM: operand matrix.
@param     	: double *vec: The operand vector.
@param		: double *res: The resulting vector.
@param		: int num: number of valid rows.
@param		: int *valid_rows: the valid rows
@return		: int: 0: fail, 1: success
remark 	        : size should be correct.
******************************************************************************/
void multiply_mtx_cer_MV(sparse *pM, double *vec, double *res, int num, int * valid_rows)
{
	int i, j, rows = pM->rows, cols, *col, v;
	double *val, result;
	for(i=0;i<num;i++, valid_rows++)
	{
		v = *valid_rows;
		cols = pM->ncols[v];
		result = pM->val[v].diag*vec[v];
		if(cols!=0)
		{
			col = pM->val[v].col; val = pM->val[v].val;
			for(j=0;j<cols;j++)
				result+=vec[*col++]*(*val++);
		}
		res[v]=result;
	}
}

/*****************************************************************************
name		: multiply_cer_MV
role       	: multiply certain rows of a matrix with a vector.
@param     	: double *vec: The operand vector.
@param		: double *res: The resulting vector.
@param		: int num: number of valid rows.
@param		: int *valid_rows: the valid rows
@return		: int: 0: fail, 1: success
remark 	        : size should be correct.
******************************************************************************/
unsigned int multiply_cer_MV(double *vec, double *res, int num, int * valid_rows)
{
	multiply_mtx_cer_MV(s, vec, res, num, valid_rows);
	return 1;
}

/*****************************************************************************
name		: multiplyTMV
role       	: multiply a vector to a matrix.
@param		: sparse * pM the matrix to be used
@param     	: double *vec: The operand vector.
@param		: double *res: The resulting vector.
@return		: double *: the resulting vector is also returned here for convinient usage
remark 	        : size should be correct.
******************************************************************************/
double * multiply_mtx_TMV(sparse * pM, double *vec, double *res)
{
	int i, j, rows = pM->rows, cols, *tcol;
	double  *tval, vec_i;
	/*Set the resulting array to zero values*/
	bzero( res, sizeof(double)*rows );
	/*Compute*/
	for( i=0; i < rows; i++ )
	{
		cols = pM->ncols[i];
		vec_i = vec[i];
		res[i] += vec_i * pM->val[i].diag;
		if(cols==0) continue;
		tcol = pM->val[i].col;
		tval = pM->val[i].val;
		for( j=0; j < cols; j++, tcol++, tval++)
			res[*tcol] += vec_i * *tval;
	}
}

/*****************************************************************************
name		: multiplyTMV
role       	: multiply a vector to a matrix.
@param     	: double *vec: The operand vector.
@param		: double *res: The resulting vector.
remark 	        : size should be correct. CHECK FOR DIAGONALS ??
******************************************************************************/
void multiplyTMV(double *vec, double *res)
{
	multiply_mtx_TMV(s, vec, res);
}                                                             

/*****************************************************************************
name		: multiply_mtx_cer_TMV
role       	: multiply a vector to a matrix.
@param		: sparse * pM the matrix to be used
@param     	: int vec: The operand vector.
@param		: int res: The resulting vector.
@return		: double *: the resulting vector is also returned here for convinient usage
@param		: int * pIds the valid ids, pIds[0] contains the number of ids
remark 	        : size should be correct.
******************************************************************************/
double * multiply_mtx_cer_TMV(sparse * pM, double * vec, double *res, int * pIds )
{
	/*ToDo: Implement*/
	int i, j, total_length = pM->rows, cols, *pCols, * pNCols = pM->ncols, length = pIds[0], id;
	double  *pVals, vec_i;
	values * pRow, * pRows = pM->val;
	
	/*Set the resulting array to zero values*/
	bzero( res, sizeof(double)*total_length );
	
	/*Compute*/
	for( i=1; i <= length; i++ )
	{
		id = pIds[i];		/*Get the valid row id*/
		cols = pNCols[id];	/*The number of columns in a row, BUG_ID: 004*/
		vec_i = vec[id];	/*The corresponding element in the vec vector*/
		pRow = & pRows[id];	/*The row values*/
		
		res[id] += vec_i * pRow->diag;	/*First multiply the diagonal value*/
		
		if( cols == 0 ) continue; /*If there are no non diagonal elements then skip*/
		
		/*Process the rest elements of the row*/
		pCols = pRow->col;
		pVals = pRow->val;
		for( j=0; j < cols; j++, pCols++, pVals++)
			res[*pCols] += vec_i * *pVals;
	}
	return res;
}

/*****************************************************************************
name		: read_file
role       	: reads the file puts in a new matrix. See remarks for format.
@param     	: char *filename: The name of the input file.
@return         : the matrix size (i.e. number of states)
remark 	        : Format of the file
		  no_of_states
		  (src_state dest_state value)*
******************************************************************************/
unsigned int read_file(char *filename)
{
        FILE *p;
        char  s[1024];
	int size, row, col;
	double val = 0.0;
	p = fopen(filename, "r");
	if(p==NULL)
	{
		printf("EROOR: Could not open .tra file \"%s\".\n",filename);
		exit(1);
	}

	if(fgets( s, 1024, p )!=NULL)
	{
		sscanf( s, "%d", &size );
		init_matrix(size,size);
	    	while (NULL != fgets( s, 1024, p ))
    		{
       	        	sscanf( s, "%d%d%lf", &row, &col, &val );
			set_val(row-1,col-1,val);
    		}
    	}	      
    	(void)fclose(p);
    	return size;
}

/*****************************************************************************
name		: split_A_into_DI_LU
role		: this method splits the given A matrix into two matrixes DI and L+U
		: where DI is an inverted diagonal matrix and L+U are all the rest elements of
		  the initial matrix A.
		  NOTE: This method doesn't need the matrix to be set with the
		  set_sparse(sparse *) method it works with the patrix pA.
@param 		: sparse *pA the initial matrix A
@param 		: sparse *pDI the empty matrix which will be filled with the D elements
@param 		: sparse *pLU the empty matrix which will be filled with the LU elements
*****************************************************************************/
void split_A_into_DI_LU(sparse* pA, sparse *pDI, sparse *pLU)
{
	int i;
	values * pD_Val = pDI->val, * pA_Val = pA->val, * pLU_Val = pLU->val;
	int * pA_Ncols = pA->ncols, * pLU_Ncols = pLU->ncols;
	for(i=0; i < pA->rows; i++)
	{
		/*Copy values to the D matrix*/
		if( pA_Val[i].diag != 0 )
			pD_Val[i].diag = 1/pA_Val[i].diag;
		
		/*Copy values to the LU matrix*/
		pLU_Ncols[i] = pA_Ncols[i];
		pLU_Val[i].col = pA_Val[i].col;
		pLU_Val[i].val = pA_Val[i].val;
	}
}

/*****************************************************************************
name		: split_A_into_D_LU
role		: this method splits the given A matrix into two matrixes DI and L+U
		: where D is a diagonal matrix and L+U are all the rest elements of
		  the initial matrix A.
		  NOTE: This method doesn't need the matrix to be set with the
		  set_sparse(sparse *) method it works with the patrix pA.
@param 		: sparse *pA the initial matrix A
@param 		: double *pD the empty vector which will be filled with the diagonal elements
@param 		: sparse *pLU the empty matrix which will be filled with the LU elements
*****************************************************************************/
void split_A_into_D_LU(sparse* pA, double *pD, sparse *pLU)
{
	int i;
	values * pA_Val = pA->val, * pLU_Val = pLU->val;
	int * pA_Ncols = pA->ncols, * pLU_Ncols = pLU->ncols;
	for( i=0; i < pA->rows; i++ )
	{
		/*Copy values to the D vector*/
		pD[i] = pA_Val[i].diag;
		
		/*Copy values to the LU matrix*/
		pLU_Ncols[i] = pA_Ncols[i];
		pLU_Val[i].col = pA_Val[i].col;
		pLU_Val[i].val = pA_Val[i].val;
	}
}

/*****************************************************************************
name		: split_A_into_D_L_U
role		: This method splits the given A matrix into two matrixes DI and L+U
		: where D is a diagonal matrix and L+U are all the rest elements of
		  the initial matrix A.
		  NOTE: This method presumes that row elements are ordered by column id
		  and that is why it does not copy the pA elements to pL, pU but
		  simply copy pointers. Take this in mind while deleting pL and pU
		  NOTE: This method doesn't need the matrix to be set with the
		  set_sparse(sparse *) method it works with the matrix pA.
@param 		: sparse *pA the initial matrix A
@param 		: double *pD the empty vector which will be filled with the diagonal elements
@param 		: sparse *pL the empty matrix which will be filled with the L elements
@param 		: sparse *pU the empty matrix which will be filled with the U elements
*****************************************************************************/
void split_A_into_D_L_U(sparse* pA, double *pD, sparse *pL, sparse *pU)
{
	short isRFirst;
	int i, j, theNColsL, theNColsU, col;
	double val;
	
	for(i=0; i < pA->rows; i++)
	{
		/*Copy values to the pD vector*/
		pD[i] = pA->val[i].diag;
		
		/*Initialize the counter for the number of row elements of the pL and pU matrixes*/
		theNColsL = 0;
		theNColsU = 0;
		
		isRFirst = 1;
		
		/*Store possible Low diagonal values, even if there are no values
		then the number of columns will be zerro and thus this will not
		affect the entire process*/
		pL->val[i].col = pA->val[i].col;
		pL->val[i].val = pA->val[i].val;
		
		/*Copy values to the pL, pU matrixes*/
		for(j = 0; j < pA->ncols[i]; j++)
		{
			/*Get the column index of the next element in a row*/
			col = pA->val[i].col[j];
			/*Check and store element in pL or in pU*/
			if( col < i )
			{
				theNColsL++;
			}
			else
			{
				/* Case of col > i*/
				if( isRFirst )
				{
					/*Copy to the pU matrix*/
					pU->val[i].col = & pA->val[i].col[j];
					pU->val[i].val = & pA->val[i].val[j];
					isRFirst = 0;
				}
				theNColsU++;
			}
		}
		pL->ncols[i] = theNColsL;
		pU->ncols[i] = theNColsU;
	}
}

/*****************************************************************************
name		: sum_vec_vec
role		: This method is used to compute the sum of two vectors A+B=Res.
@param		: int length the length of vectors
@param 		: double *pA vector A
@param 		: double *pB vector B
@param 		: double *pRes vector Res
*****************************************************************************/
double * sum_vec_vec(int length , double * pA, double * pB, double * pRes )
{
	int i;
	for( i = 0; i < length; i++ )
	{
		pRes[i] = pB[i] + pA[i];
	}
	return pRes;
}


/*****************************************************************************
name		: sum_vec_vec
role		: This method is used to compute the sum of two vectors A+B=Res.
@param 		: double *pA vector A
@param 		: double *pB vector B
@param 		: double *pRes vector Res
@param		: int * pIds the valid ids, pIds[0] contains the number of ids
*****************************************************************************/
double * sum_cer_vec_vec(double * pA, double * pB, double * pRes, int * pIds )
{
	int i, length = pIds[0], id;
	for( i = 1; i <= length ; i++ )
	{
		id = pIds[i];
		pRes[ id ] = pB[ id ] + pA[ id ];
	}
	return pRes;
}

void print_vec_double(int length, double * vec)
{
	int i;
	printf("( ");
	for( i = 0 ; i < length ; i++ )
	{
		if( vec[i] == 0.0 || vec[i] == 1.0 )
		{
			printf( "%1.1le", vec[i] );
		}else{
			printf( "%1.15le", vec[i] );
		}
		if (i != length-1) printf(", ");
	}
	printf(" )\n");
}

void print_vec_int(int length, int * vec)
{
	int i;
	printf("( ");
	for( i = 0 ; i < length ; i++ )
	{
		printf( "%d", vec[i] );
		if (i != length-1) printf(", ");
	}
	printf(" )\n");
}

void print_vec_short(int length, short * vec)
{
	int i;
	printf("( ");
	for( i = 0 ; i < length ; i++ )
	{
		printf( "%d", vec[i] );
		if (i != length-1) printf(", ");
	}
	printf(" )\n");
}

/*****************************************************************************
name		: multiply_inv_diag_D_V
role       	: multiply v*diag(d)I. I.e. take vector of diagonal elements
		  create matrix diag(d) invert it and multiply by the vector v
		  from the left side.
@param     	: double * pD: The diagonal elements.
@param     	: double * pV: The vector to multiply with.
@param     	: double * pR: The resulting vector.
@param     	: int length : The length of vectors
******************************************************************************/
void multiply_inv_diag_D_V(double * pD, double * pV, double * pR, const int length)
{
	int i;
	double diag;
	for( i = 0; i < length; i++ )
	{
		diag = pD[i];
		if( diag != 0 )
		{
			pR[i] = pV[i]/diag;
		}
	}
}

/*****************************************************************************
name		: multiply_cer_inv_diag_D_V
role       	: multiply v*diag(d)I. I.e. take vector of diagonal elements
		  create matrix diag(d) invert it and multiply by the vector v
		  from the left side.
@param     	: double * pD: The diagonal elements.
@param     	: double * pV: The vector to multiply with.
@param     	: double * pR: The resulting vector.
@param     	: int * pValidStates : The set of valid states
******************************************************************************/
void multiply_cer_inv_diag_D_V(double * pD, double * pV, double * pR, int * pValidStates)
{
	int i,idx;
	double diag;
	const int length = pValidStates[0];
	for( i = 1; i <= length; i++ )
	{
		idx = pValidStates[i];
		diag = pD[idx];
		if( diag != 0 )
		{
			pR[idx] = pV[idx]/diag;
		}
	}
}

