/**
*	WARNING: Do Not Remove This Section
*
*	$RCSfile: transient.c,v $
*	$Revision: 1.124 $
*	$Date: 2006/02/14 12:22:15 $
*	$Author: zapreevis $
*	
*	MRMC is a model checker for discrete-time and continuous-time Markov reward models.
*	It supports reward extensions of PCTL and CSL (PRCTL and CSRL), and allows for the
*	automated verification of properties concerning long-run and instantaneous rewards
*	as well as cumulative rewards.
*	
*	Copyright (C) The University of Twente, 2004-2006.
*	Authors: Maneesh Khattri, Ivan Zapreev
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*	Our contacts:
*		Formal Methods and Tools Group, University of Twente,
*		P.O. Box 217, 7500 AE Enschede, The Netherlands,
*		Phone: +31 53 4893767, Fax: +31 53 4893247,
*		E-mail: mrmc@cs.utwente.nl
*
*	Source description: Perform Transient Analysis for PCTL/CSL/PRCTL/CSRL - X, U.
*	Uses: DEF: sparse.h, bitset.h, label.h, runtime.h, transient.h, foxglynn.h
*		LIB: sparse.c, bitset.c, label.c, runtime.c, transient.c, foxglynn.c
*	Remarks: Perform Transient Analysis for PCTL/CSL/PRCTL/CSRL  - X, U.
*/

# include "macro.h"
#ifdef MAC_OS
	# include <stdlib.h>
	# include <strings.h>
#else
	# include <malloc.h>
#endif
#include <math.h>
#include "foxglynn.h"
#include "bitset.h"
#include "sparse.h"
#include "label.h"
#include "runtime.h"
#include "iterative_solvers.h"
#include "path_graph.h"
#include "discretization.h"
#include "uniformization_sericola.h"
#include "uniformization_qureshi_sanders.h"
#include "transient.h"
#include "bscc.h"
#include <sys/types.h>

/*****************************************************************************
name		: get_exist_until
role       	: solve E(phi U psi) until formula.
@param		: sparse *state_space: the state space
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: bitset *psi: satisfaction relation for psi formula.
@return		: bitset *: result of E(SAT(phi) U SAT(psi)) for all states.
remark 	        : adapted for t-bounded-until from
	  1. F. Ciesinski, M. Grober. On Probabilistic Computation Tree Logic.
	     LNCS, Vol. 2925, Springer, pp. 147-188, 2004.
******************************************************************************/
static bitset * get_exist_until(sparse *state_space, bitset *phi, bitset *psi)
{
	int i, j, size = phi->n, reach=0, pcols;
	double val;
	int *back_set;
	int * states = NULL;
	bit get1b, get2b;
	bitset *EU = or(get_new_bitset(size), psi);
	set_sparse(state_space);
	for(i=0;i<size;i++)
	{ 
		get_bit_val(psi, i, &get1b);
		if(get1b.b)
		{
			states = (int *) realloc(states, (reach+1)*sizeof(int));
			states[reach++] = i;
		}
	}
	for(i=0;i<reach;i++)
	{
		pcols = state_space->pcols[states[i]];
		back_set = state_space->val[states[i]].back_set;
		for(j=0;j<pcols;j++)
		{
			get_bit_val(phi, back_set[j], &get1b);
			get_bit_val(EU, back_set[j], &get2b);
			if(get1b.b && !get2b.b)
			{
				states = (int *) realloc(states, (reach+1)*sizeof(int));
				states[reach++] = back_set[j];
				set_bit_val(EU, back_set[j], &get1b);
			}
		}
	}
	free(states); states=NULL;
	return EU;
}

/*****************************************************************************
name		: get_always_until
role       	: solve A(phi U psi) until formula.
@param		: sparse *state_space: the state space
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: bitset *psi: satisfaction relation for psi formula.
@return		: bitset *: result of A(SAT(phi) U SAT(psi)) for all states.
remark 	        : adapted for t-bounded-until from
		  1. F. Ciesinski, M. Grober. On Probabilistic Computation Tree Logic.
		     LNCS, Vol. 2925, Springer, pp. 147-188, 2004.
******************************************************************************/
static bitset * get_always_until(sparse *state_space, bitset *phi, bitset *psi, bitset *e_phi_psi)
{
	int i, j, k, l, size = phi->n, reach=0, ncols, *col=NULL, pcols, *back_set=NULL, *toremove=NULL, pres_size, flag;
	double val;
	bit get1b, get2b, reset1b;
	bitset *AU = or(get_new_bitset(size), e_phi_psi);
	set_sparse(state_space);
	reset1b.b = 0;
	/* find a state in E(Phi U Psi) that does not satisfy A(Phi U Psi) */
	for(i=0;i<size;i++)
	{ 
		flag=0;
		get_bit_val(AU, i, &get1b);
		get_bit_val(psi, i, &get2b);
		if(get1b.b&&!get2b.b)
		{
			ncols = state_space->ncols[i];
			col = state_space->val[i].col;
			for(j=0;j<ncols;j++)
			{
				get_bit_val(AU, col[j], &get2b);
				if(!get2b.b)
					{flag=1; break;}
			}
			
			if(flag)
			{
				/* remove i and its predecessors */
				pres_size = 1;
				toremove = (int *)calloc(pres_size, sizeof(int));
				toremove[0] = i;
				set_bit_val(AU, toremove[pres_size-1], &reset1b);
				for(k=0;k<pres_size;k++)
				{
					pcols = state_space->pcols[toremove[k]];
					back_set = state_space->val[toremove[k]].back_set;
					/* schedule predecessors for removal */
					for(l=0;l<pcols;l++)
					{
						get_bit_val(AU, back_set[l], &get1b);
						get_bit_val(psi, back_set[l], &get2b);
						if(get1b.b&&!get2b.b)
						{
							toremove=(int *)realloc(toremove, (pres_size+1)*sizeof(int));
							toremove[pres_size++] = back_set[l];
							set_bit_val(AU, toremove[pres_size-1], &reset1b);
						}
					}				
				}
				free(toremove);
				toremove = NULL;
			}
		}
	}
	return AU;
}

/*****************************************************************************
name		: ab_state_space
role       	: make certain states (not in n_absorbing) absorbing.
			  Creates a new empty sparse matrix, then assighs non absorbing rows via pointers
			  and computes uniformization rates for them.
@param		: sparse *state_space: the state space
@param		: bitset *n_absorbing: not absorbing states.
@param		: double *lambda(ref.): return value of uniformizing rate(also q).
@param		: double *abse: return row_sums
@return		: sparse *: new sparse matrix with some states made absorbing.
remark 	        : makes states absorbing (in ab_state_space) by assigning pointers
		  to rows (in state_space) only for those rows which are not to be
		  made absorbing.
		  NOTE: initially *lambda should be equal to 0!!!!
******************************************************************************/
sparse * ab_state_space(sparse * state_space, bitset *n_absorbing, double *lambda, double *abse, int *non_absorbing)
{
	if ( *lambda != 0.0 )
	{
		printf("ERROR: lambda parameter in ab_state_space is not 0!");
		exit(1);
	}
	int i, n=n_absorbing->n;
	bit getb;
	sparse *ab_state_space; 
	double *e = get_e(); /*Get row sums of the initial matrix*/
	init_matrix(state_space->rows, state_space->cols); /*Init a new matrix structure*/
	for( i=0; i<n; i++ )  /*For all not absorbing states*/
	{
		get_bit_val(n_absorbing,i,&getb);
		if( getb.b )
		{
			++*non_absorbing;
			if((*lambda)<e[i]) (*lambda)=e[i]; /*Get max among available row sums*/
			abse[i] = e[i];
			set_row(i, state_space->ncols[i], state_space->val[i]); /*Assign rows via pointers*/
		}
	}
	if( *lambda > 1.0 ) ++(*lambda); /*Make uniformization rate strictly > than max row sum*/
	ab_state_space = get_sparse();
	return ab_state_space;
}

/*****************************************************************************
name		: ab_state_space_ssd
role       	: make certain states (not in n_absorbing) absorbing.
			  Creates a new empty sparse matrix, then assighs non absorbing rows via pointers
			  and computes uniformization rates for them. This method removes from n_absorbing
			  phi states which belong to BSCCs. So the steady state detection in uniformization
			  can be used.
@param		: sparse *state_space: the state space
@param		: bitset *n_absorbing: not absorbing states, all of them are phi states, is modified
              in this method.
@param		: double *lambda(ref.): return value of uniformizing rate(also q).
@param		: double *abse: return row_sums
@return		: sparse *: new sparse matrix with some states made absorbing.
remark      : makes states absorbing (in ab_state_space) by assigning pointers
		      to rows (in state_space) only for those rows which are not to be
	          made absorbing.
              NOTE: initially *lambda should be equal to 0!!!!
******************************************************************************/
static sparse * ab_state_space_ssd(sparse * state_space, bitset *n_absorbing, double *lambda, double *abse, int *non_absorbing)
{
  /*Make not phi or psi state absorbing*/
  sparse * abs_npho_psi = ab_state_space(state_space, n_absorbing, lambda, abse, non_absorbing);
  /*Store state space dimensions*/
  const int size = abs_npho_psi->rows;
  /*Create a TBSCCs structure to search for phi states that belong to BSCCs in abs_npho_psi*/
  TBSCCs * pBSCCsHolder = allocateTBSCCs(abs_npho_psi);
  /*Search for BSCCs among n_absorbing states, all of them are phi states*/
  int ** ppNewBSCCs = getNewBSCCs(pBSCCsHolder, n_absorbing);

  short * pbscc_mapping = getStatesBSCCsMapping(pBSCCsHolder);
  bit getb;
  int i;
  /*Removes phi states which belong to some bscc from n_absorbing*/
  for( i=0; i<size; i++ )
  {
    get_bit_val(n_absorbing,i,&getb);
    if( getb.b && pbscc_mapping[i] != 0) /*If a non absorbing phi state is in some bscc*/
    {
	  set_bit_val(n_absorbing,i,&BIT_OFF); /*Remove this state as it should become absorbing*/
    }
  }
  /*Now n_absorbing contains those states that are phi and do not belong to any bscc*/
  
  /*Free temporary structures*/
  free_abs(abs_npho_psi);
  freeTBSCC(pBSCCsHolder);
  freeBSCCs(ppNewBSCCs);

  /*Set counter of not absorbing states to 0 as soon as we will
    recalculte this in the next call of ab_state_space*/
  *non_absorbing = 0;
  /*It should be 0 in the call of ab_state_space*/
  *lambda = 0.0;
  /*Clean abse as we will have less values, and they will be set
    in the call of ab_state_space*/
  bzero( abse, sizeof(double)*size );

  /*Make phi states that belong to BSCCs in abs_npho_psi absorbing*/
  return ab_state_space(state_space, n_absorbing, lambda, abse, non_absorbing);
}

/**************************pM->rows*********************************************
name		: free_abs
role       	: free absorbing sparse matrix.
@param		: sparse *abs: the state space
remark 	    : It's row elements were copied as structures,
			  so the internal row (values structure) pointers refer to the original matrix
			  and thus should not be cleaned !!! 
******************************************************************************/
void free_abs(sparse * abs)
{
	free(abs->ncols);
	free(abs->pcols);
	free(abs->val); //Do not clean each row, see the remark
	free(abs);
}

/*****************************************************************************
name		: count_set
role       	: count the number of valid bits in a bitset.
		  put all the valid-bit ids in an int-array.
		  the first element of the array is the count.
		  the rest are the bit ids of valid (set) bits.
@param		: bitset *toCount: the bitset in question
@return		: the count of bits that are set and their ids.
remark 	        : 
******************************************************************************/
static int * count_set(bitset *toCount)
{
	int size = toCount->n, i;
	bit getb;
	int *pValidSet = (int *) calloc (1, sizeof(int));
	pValidSet[0] = 0;
	for(i=0;i<size;i++)
	{
		get_bit_val(toCount, i, &getb);
		if(getb.b)
		{
			++pValidSet[0];
			pValidSet = (int *) realloc (pValidSet, (pValidSet[0]+1) * sizeof(int));
			pValidSet[pValidSet[0]] = i;
		}
	}
	return pValidSet;
}

/*****************************************************************************
name		: unbounded_until
role       	: solve the unbounded until operator for all states.
@param		: bitset *phi: SAT(phi).
@param		: bitset *psi: SAT(psi).
@return		: double *: result of the unbounded until operator for all states.
remark 	        : adapted from
	  1. F. Ciesinski, M. Grober. On Probabilistic Computation Tree Logic.
		     LNCS, Vol. 2925, Springer, pp. 147-188, 2004.
******************************************************************************/
static double * unbounded_until(bitset *phi, bitset *psi)
{
	sparse *state_space = get_state_space();
	bitset * EU = get_exist_until(state_space, phi, psi);
	bitset * AU = get_always_until(state_space, phi, psi, EU);
	int i, size = state_space->rows;
	double * initial = (double *)calloc(size,sizeof(double));
	double *rhs = (double *)calloc(size,sizeof(double)), *result=NULL;
	double err = get_error_bound();
	int max_iterations = get_max_iterations();
	bit getb;
	sparse *pQ = init_matrix(size,size);
	bitset *dummy=NULL;
	int *pValidStates=NULL;
	
	dummy = not(EU);
	or_result(AU, dummy);
	not_result(dummy);
	pValidStates=count_set(dummy);
	/* make ~(Phi /\ ~Psi) states absorbing */
	initMatrix(state_space, pQ, pValidStates);
	/* get the embedded DTMC P */
	mult_mtx_cer_const_array(pQ, get_e(), pValidStates, 1);
	/* get (I-P) */
	mult_mtx_cer_const(pQ, -1, pValidStates);
	add_cer_cons_diagonal_mtx(pQ, pValidStates, 1);
	for(i=0;i<size;i++)
	{
		get_bit_val(AU,i,&getb);
		if(getb.b){ rhs[i] = 1.0; initial[i] = 1.0; }
	}
	/* solve (I-P)x = 1_Psi */
	result = solveGaussJacobi(pQ, initial, rhs, err, max_iterations, pValidStates, 1);
	/* revert matrix P -> Q 
	Note: not required for diagonal elements */
	mult_mtx_cer_const(pQ, -1, pValidStates);
	mult_mtx_cer_const_array(pQ, get_e(), pValidStates, 0);
	cleanMatrix(pQ, pValidStates);
	free(pQ);
	free(dummy);
	free(pValidStates);
	return result;
}

/*****************************************************************************
name		: uniformization_plain
role       	: solve the bounded until operator by uniformization, without steady-state detection
@param		: bitset *n_absorbing: non-absorbing states.
@param		: bitset *reach: goal states for instance SAT(psi), i.e the i_\psi vector.
@param		: float supi: sup I
@return		: double *: result of the unbounded until operator for all states.
remark 	        : see:
	  1. J.-P. Katoen, M.Z. Kwiatowska, G. Norman, D.A. Parker.
	  Faster and symbolic CTMC model checking. LNCS Vol. 2165, Springer,
	  Aachen, Germany, pp. 23-38, 2001.
******************************************************************************/
static double * uniformization_plain(bitset *n_absorbing, double *reach, float supi)
{
  sparse *state_space = get_state_space();
  const double eps=get_error_bound();
  const double u=get_underflow();
  const double o=get_overflow();
  const int size = state_space->rows;
  /*Store the initial pointer to the reach as it will be freed somewhere else*/
  const double * golden_reache = reach;
  
  double lambda=0, current_fg=0.0;
  int i, j, k, non_absorbing=0;

  /*diag is used to store diagonal values from the abs matrix*/
  double * diag = (double *)calloc(size, sizeof(double));
  /*res is used to store iterates*/
  double * res = (double *)calloc(size, sizeof(double));
  /*result is used to store partial sum*/
  double * result = (double *)calloc(size, sizeof(double));

  /*Make states absorbing: not phi || psi*/
  sparse *abs = ab_state_space(state_space, n_absorbing, &lambda, diag, &non_absorbing);
  /*allocate the array which will hold ids of valid rows from abs matrix*/
  int * valid_rows = (int *)calloc(non_absorbing, sizeof(int));

  //print_mtx_sparse(state_space);
  //print_mtx_sparse(abs);

  /*Find valid rows*/
  bit getb;
  for(i=0, j=0; i<size; i++)
  {
	get_bit_val(n_absorbing,i,&getb);
    if(getb.b) valid_rows[j++]=i;
  }
  /*This copying is vital because we need to preserve 1.0 values of absorbing states*/
  memcpy( res, reach, sizeof(double)*size );

  /*Initially, vectors res = reach!*/
  FoxGlynn fg;
  if( fox_glynn(lambda*supi, u, o, eps, &fg) )
  {
    printf("Fox-Glynn: ltp = %d, rtp = %d, w = %1.15le\n",fg.left,fg.right,fg.total_weight);

    /*R-E(s)*/
    sub_mtx_diagonal(abs, diag);
    /*Uniformize : lambda>0*/
    mult_mtx_const(abs, 1/lambda);
    add_mtx_cons_diagonal(abs, 1.0);

    //print_mtx_sparse(abs);  
		
    /*This variable will be used to store a tmp pointer to an array*/
     double * tmp_arr = NULL;
		
    /*Skip upto left*/
    for( i=1; i < fg.left; i++ )
    {
      multiply_mtx_cer_MV(abs, reach, res, non_absorbing, valid_rows);
      /*Flip pointers*/
      tmp_arr = reach; reach = res; res = tmp_arr;
    }

    /*Compute upto right*/
    for( ; i <= fg.right; i++ )
    {
      current_fg = fg.weights[i-fg.left];
      multiply_mtx_cer_MV(abs, reach, res, non_absorbing, valid_rows);
      for( j=0; j < size; j++ )
        result[j] += current_fg*res[j];
      /*Flip pointers*/
      tmp_arr = reach; reach = res; res = tmp_arr;
    }
    /*Divide with total weight*/
    for( j=0; j < size; j++ ) result[j] /= fg.total_weight;

    /*Reset the matrix to its original state
      NOTE: operations on diagonals are not required */
    mult_mtx_const(abs, lambda);
  }

  /*Free dummy vectors*/
  free( valid_rows );
  free( diag );
  /*Free an array allocated here but not in the external function*/
  if( golden_reache == res )
    free( reach );
  else
    free( res );
  free_abs( abs );

  return result;
}

/*****************************************************************************
name		: isSteadyState
role       	: Detects steady-state while uniformization for CSL logic.
@param		: double * reach_psi an array of probabilities to reach psi states
              for some iteration
@param		: double * reach_bad an array of probabilities to reach
             (not phi and not psi) U (phi state in some bscc) states
              for some iteration
@param      : const double delta the error threshold
@param		: const int valid_size the length valid_states array
@param		: int* valid_states array with ids of transient states
@return		: TRUE if steady state is reached, FALSE otherwise
remark 	    : We use the property that the probabilityh to be in transient state
              becomes 0 in the long run
******************************************************************************/
static BOOL isSteadyState(double * reach_psi, double * reach_bad, const double delta,
                          const int valid_size, int* valid_states)
{
  int i = 0;

  //printf("isSSD?:\n");
  //print_vec_double(get_state_space()->rows, reach_psi);
  //print_vec_double(get_state_space()->rows, reach_bad);

  for( i=0; i < valid_size; i++)
  {
    /*Test the probability to remane in transient states when starting in state i*/
    if( ( 1 - reach_psi[valid_states[i]] - reach_bad[valid_states[i]] ) > delta)
	  return FALSE;
  }

  return TRUE;
}

/*****************************************************************************
name		: uniformization_ssd
role       	: solve the bounded until operator by uniformization with steady-state detection
@param		: bitset *n_absorbing: non-absorbing states.
@param		: bitset *psi : psi states for steady-state detection
@param		: bitset *reach_psi: goal states for instance SAT(psi), i.e the i_\psi vector.
@param		: float supi: sup I
@return		: double *: result of the unbounded until operator for all states.
remark 	        : see:
	  1. J.-P. Katoen, M.Z. Kwiatowska, G. Norman, D.A. Parker.
	  Faster and symbolic CTMC model checking. LNCS Vol. 2165, Springer,
	  Aachen, Germany, pp. 23-38, 2001.
******************************************************************************/
static double * uniformization_ssd(bitset *n_absorbing, bitset *psi, double *reach_psi, float supi)
{
  sparse *state_space = get_state_space();
  const double eps=get_error_bound();
  const double u=get_underflow();
  const double o=get_overflow();
  const int size = state_space->rows;
  /*Store the initial pointer to the reach_psi as it will be freed somewhere else*/
  const double * golden_reache_psi = reach_psi;
  
  double lambda=0, current_fg=0.0;
  int i, j, k, non_absorbing=0;

  /*diag is used to store diagonal values from the abs matrix*/
  double * diag = (double *)calloc(size, sizeof(double));
  /*res_psi is used to store iterates res_psi = P*reach_psi*/
  double * res_psi = (double *)calloc(size, sizeof(double));
  /*res_bad is used to store iterates, res_bad = P*reach_bad*/
  double * res_bad = (double *)calloc(size, sizeof(double));
  /*reach_bad is used to store iterates*/
  double * reach_bad = (double *)calloc(size, sizeof(double));
  /*result is used to store partial sum*/
  double * result = (double *)calloc(size, sizeof(double));

  /*Make states absorbing: not phi || psi, and then those phi that will be remaining in some bscss*/
  sparse *abs = ab_state_space_ssd(state_space, n_absorbing, &lambda, diag, &non_absorbing);
  /*allocate the array which will hold ids of valid rows from abs matrix*/
  int * valid_rows = (int *)calloc(non_absorbing, sizeof(int));

  /*Construct the initial vector for the absorbing states which are not Psi states, i.e. bad_states*/
  bitset * bad_states = or(n_absorbing,psi);
  not_result(bad_states);

  /*Find valid rows and also create an initial vector for bad states*/
  bit getb, getb_bad;
  for( i=0, j=0; i<size; i++ )
  {
    /*Find valid rows*/
    get_bit_val(n_absorbing,i,&getb);
    if( getb.b )
      valid_rows[j++]=i;
	/*Create an initial vector for bad states*/
	get_bit_val(bad_states,i,&getb_bad);
	if( getb_bad.b )
	  reach_bad[i] = 1.0;
  }
  /*This copying is vital because we need to preserve 1.0 values of absorbing states*/
  memcpy( res_bad, reach_bad, sizeof(double)*size );
  memcpy( res_psi, reach_psi, sizeof(double)*size );
  
  /*Initially, vectors res = reach!*/
  FoxGlynn fg;
  /*Note that for steady-state detection we need to take error bound
    eps/2 for the Fox-Glynn algorithm*/
  if( fox_glynn(lambda*supi, u, o, eps / 2.0, &fg) )
  {
    printf("Fox-Glynn: ltp = %d, rtp = %d, w = %1.15le\n",fg.left,fg.right,fg.total_weight);

    /* R-E(s) */
    sub_mtx_diagonal(abs,diag);
    /* uniformize : lambda>0 */
    mult_mtx_const(abs, 1/lambda);
    add_mtx_cons_diagonal(abs, 1.0);
	
    //print_mtx_sparse(abs);

    /*becomes TRUE when the steady-state is reached*/
    BOOL isSS = FALSE;
    const double delta = eps / 4.0;
    /*This variable will be used to store a tmp pointer to an array*/
    double * tmp_arr = NULL;
    /*This is a magic M constant, we check for ssd every M iterations*/
    const int M = 10;

    /*Skip upto left trancation point*/		
    for( i=1; i < fg.left; i++ )
    {
      multiply_mtx_cer_MV(abs, reach_psi, res_psi, non_absorbing, valid_rows);
      multiply_mtx_cer_MV(abs, reach_bad, res_bad, non_absorbing, valid_rows);
      if( ( i % M == 0 ) && ( isSS = isSteadyState(res_psi, res_bad, delta, non_absorbing, valid_rows) ) )
      {
        /*The steady-state has been reached before the left truncation point*/
        memcpy( result, res_psi, sizeof(double)*size );
        printf("SSD: i = %d < fg.left\n", i);
        break;
      }
      /*Flip pointers*/
      tmp_arr = reach_psi; reach_psi = res_psi; res_psi = tmp_arr;
      tmp_arr = reach_bad; reach_bad = res_bad; res_bad = tmp_arr;
    }
    
    /*If the steady state has not been reached, then compute upto right truncation point*/
    if( !isSS )
    {
      double part_sum = 0; /*will hold sum of poisson probs from left truncation point to steady-state point*/
      for( ; i <= fg.right; i++ )
      {
        current_fg = fg.weights[i-fg.left];
        multiply_mtx_cer_MV(abs, reach_psi, res_psi, non_absorbing, valid_rows);
        multiply_mtx_cer_MV(abs, reach_bad, res_bad, non_absorbing, valid_rows);
        for( j=0; j < size; j++ )
          result[j] += current_fg*res_psi[j];
        part_sum += current_fg; /*Accumulate weights, because the SSD may be reached here, then we will need them*/
        if( ( i % M == 0 ) && ( isSS = isSteadyState(res_psi, res_bad, delta, non_absorbing, valid_rows) ) )
        {
          printf("SSD: i = %d, part_sum = %1.15le\n", i, part_sum);
          break;
        }
        /*Flip pointers*/
        tmp_arr = reach_psi; reach_psi = res_psi; res_psi = tmp_arr;
        tmp_arr = reach_bad; reach_bad = res_bad; res_bad = tmp_arr;
      }
	  
      /*The part after the steady-state point*/
      if( isSS )
      {
        double mult = fg.total_weight - part_sum;
        for( j=0; j < size; j++ ) result[j] += res_psi[j] * mult;
      }
      /*Divide with total weight */
      for( j=0; j < size; j++ ) result[j] /= fg.total_weight;
    }
    /*Reset the matrix to its original state
      NOTE: operations on diagonals are not required */
    mult_mtx_const(abs,lambda);
  }

  /*Free dummy vectors*/
  free( valid_rows );
  free( diag );
  /*Free an array allocated here but not in the external function*/
  if( golden_reache_psi == res_psi )
    free( reach_psi );
  else
    free( res_psi );
  free( res_bad );
  free( reach_bad );
  free_bitset( bad_states );
  free_abs( abs );

  return result;
}

/*****************************************************************************
name		: uniformization
role       	: solve the bounded until operator by uniformization, this method is a wrapper
              depending on setting it either calls a plain uniformization or uses uniformization
			  with a steady-state detection
@param		: bitset *n_absorbing: non-absorbing states.
@param		: bitset * psi: psi states for steady-state detection, may be NULL if (ssd_allowed == FALSE)
@param		: bitset *reach: goal states for instance SAT(psi), i.e the i_\psi vector.
@param		: float supi: sup I
@param		: BOOL ssd_allowed: true if steady state detection is allowed to be used, it does not
			  mean that it will be used though, the later depends on runtime settings, by default
			  ssd it is off.
@return		: double *: result of the unbounded until operator for all states.
remark 	        : see:
	  1. J.-P. Katoen, M.Z. Kwiatowska, G. Norman, D.A. Parker.
	  Faster and symbolic CTMC model checking. LNCS Vol. 2165, Springer,
	  Aachen, Germany, pp. 23-38, 2001.
******************************************************************************/
static double * uniformization(bitset *n_absorbing, bitset *psi, double *reach, float supi, BOOL ssd_allowed)
{
  if( ssd_allowed && is_ssd_on() )
  {
	if(psi == NULL)
	{
	  printf("ERROR: The steady-state detection is on, but psi states (for phi U[0,t] psi) are not set.\n");
	  exit(1);
	}
	return uniformization_ssd(n_absorbing, psi, reach, supi);
  }else{
	return uniformization_plain(n_absorbing, reach, supi);
  }
}

/*****************************************************************************
name		: bounded_until
role       	: solve the bounded until operator.
@param		: bitset *phi: SAT(phi).
@param		: bitset *psi: SAT(psi).
@param		: float supi: sup I
@return		: double *: result of the unbounded until operator for all states.
remark 	        : see:
	  1. J.-P. Katoen, M.Z. Kwiatowska, G. Norman, D.A. Parker.
	  Faster and symbolic CTMC model checking. LNCS Vol. 2165, Springer,
	  Aachen, Germany, pp. 23-38, 2001.
******************************************************************************/
static double * bounded_until(bitset *phi, bitset *psi, float supi)
{
	sparse * state_space = get_state_space();
	int i, size = state_space->rows;
	double *reach = (double *)calloc(state_space->rows, sizeof (double)), *result;
	bit getb;
	/*bitset * EU = get_exist_until(state_space, phi, psi);
	bitset * AU = get_always_until(state_space, phi, psi, EU);*/
	bitset *dummy=not(psi);
	/*dummy = states which are not absorbing i.e. not(Psi or not Phi) = not Psi and Phi*/
	dummy = and(phi, dummy);
	
	for(i=0;i<size;i++)
	{
		get_bit_val(psi,i,&getb);
		if(getb.b) reach[i] = 1.0;
	}
	result=uniformization(dummy, psi, reach, supi, TRUE);
	free(dummy);
	free(reach);
	return result;
}

/*****************************************************************************
name		: interval_until
role       	: solve the interval until operator.
@param		: bitset *phi: SAT(phi).
@param		: bitset *psi: SAT(psi).
@param		: float subi: sub I
@param		: float supi: sup I
@return		: double *: result of the interval until operator for all states.
remark 	        : see:
	  1. C.Baier, B.R. Haverkort, H. Hermanns and J.-P. Katoen.
	  Model Checking Algorithms for Continuous-Time Markov Chains.
	  IEEE Transactions on Software Engineering, Vol. 29, No. 6,
	  pp. 299-316, 2000.
******************************************************************************/
static double * interval_until(bitset *phi, bitset *psi, float subi, float supi)
{
	int size = phi->n, i;
	sparse * state_space = get_state_space();
	double * reach = (double *)calloc(size, sizeof(double));
	double *result1 = NULL, *result2=NULL;	
	bit getb;
	/*bitset * EU = get_exist_until(state_space, phi, psi);
	bitset * AU = get_always_until(state_space, phi, psi, EU);*/
	bitset *dummy=not(psi);
	/*dummy = states which are not absorbing i.e. not(Psi or not Phi) = not Psi and Phi*/
	dummy = and(phi, dummy);
	
	for(i=0;i<size;i++)
	{
		get_bit_val(psi,i,&getb);
		if(getb.b) reach[i] = 1.0;
	}
	result2 = uniformization(dummy, NULL, reach, (supi-subi), FALSE);
	result1 = uniformization(phi, NULL, result2, subi, FALSE);
	
	free(result2);
	free(reach);
	free(dummy);
	return result1;
}

/**
* Creates new R matrix and copies all the rows that belong to the pValidStates into it
* @param pStateSpace the initial matrix
* @param pP the new matrix to where to copy rows
* @param pValidStates the valid states i.e. ids of the rows to be copied
*                     this array contains the number of nodes as the first element
*/
void makeAbsorbingDTMC(const sparse * pStateSpace, sparse * pP, int * pValidStates)
{
	int i;
	int * pNCols = pStateSpace->ncols;
	values * pRows = pStateSpace->val;
	int valid_state;
	
	/*Set diagoral velues to 1 for all*/
	const int size = pStateSpace->rows;
	for( i = 0; i < size  ; i++ )
	{
		pP->val[i].diag = 1;
	}
	
	/*Form the pP matrix, here the 1 values of the pValidStates get correct values*/
	for( i = 1; i <= pValidStates[0] ; i++ )
	{
		valid_state = pValidStates[i];
		set_mtx_row( pP, valid_state, pNCols[valid_state], pRows[valid_state] );
	}
}

/*****************************************************************************
name		: dtmc_unbounded_until
role       	: solve the unbounded until operator for all states for DTMC.
@param		: bitset *phi: SAT(phi).
@param		: bitset *psi: SAT(psi).
@return		: double *: result of unbounded until operator for all states.
remark 	        : see:
	  1. F. Ciesinski, M. Grober. On Probabilistic Computation Tree Logic.
	     LNCS, Vol. 2925, Springer, pp. 147-188, 2004.
******************************************************************************/
static double * dtmc_unbounded_until(bitset *phi, bitset *psi)
{
	sparse *state_space = get_state_space();
	bitset * EU = get_exist_until(state_space, phi, psi);
	bitset * AU = get_always_until(state_space, phi, psi, EU);
	int i, size = state_space->rows;
	double * initial = (double *)calloc(size,sizeof(double));
	double *rhs = (double *)calloc(size,sizeof(double)), *result=NULL;
	double err = get_error_bound();
	int max_iterations = get_max_iterations();
	bit getb;
	sparse *pP = init_matrix(size,size);
	bitset *dummy=NULL;
	int *pValidStates=NULL;
	
	dummy = not(EU);
	or_result(AU, dummy);
	not_result(dummy);
	pValidStates=count_set(dummy);
	/* make ~(Phi /\ ~Psi) states absorbing */
	makeAbsorbingDTMC(state_space, pP, pValidStates);
	
	for(i=0;i<size;i++)
	{
		get_bit_val(AU,i,&getb);
		if(getb.b){ rhs[i] = 1.0; initial[i] = 1.0; }
	}

	/* get (I-P) */
	mult_mtx_cer_const(pP, -1, pValidStates);
	add_cer_cons_diagonal_mtx(pP, pValidStates, 1);
	
	/* solve (I-P)x = i_Psi */
	result = solveGaussJacobi(pP, initial, rhs, err, max_iterations, pValidStates, 1);
	
	mult_mtx_cer_const(pP, -1, pValidStates);
	
	free_mtx_wrap(pP);
	free(dummy);
	free(pValidStates);
	
	return result;
}

/*****************************************************************************
name		: dtmc_bounded_until
role       	: solve the bounded until operator for DTMC.
@param		: bitset *phi: SAT(phi).
@param		: bitset *psi: SAT(psi).
@param		: float supi: sup I should contain the Natural number
@return		: double *: result of the unbounded until operator for all states.
remark 	        : see:
	  1. F. Ciesinski, M. Grober. On Probabilistic Computation Tree Logic.
	     LNCS, Vol. 2925, Springer, pp. 147-188, 2004.
******************************************************************************/
static double * dtmc_bounded_until(bitset *phi, bitset *psi, float supi)
{
	const sparse * state_space = get_state_space();
	sparse *pP;
	int i;
	const int size = state_space->rows;
	double *result_1 = (double *)calloc(state_space->rows, sizeof (double));
	bitset *dummy=NULL;
	double *result_2, * pTmp;
	int * pValidStates;
	
	
	/*init result_1 with i_psi*/
	bit getb;
	for(i=0;i<size;i++)
	{
		get_bit_val(psi,i,&getb);
		if(getb.b) result_1[i] = 1.0;
	}
	
	/*Make ~(Phi /\ ~Psi) states absorbing*/
	dummy = not(phi);
	or_result(psi, dummy);
	not_result(dummy);
	pValidStates = count_set(dummy);
	
	pP = init_matrix(size,size);
	makeAbsorbingDTMC(state_space, pP, pValidStates);
	
	/*Compute Q^supi*i_psi*/
	result_2 = (double *)calloc(size, sizeof (double)), pTmp = NULL;
	for(i = 1; i <= supi ; i++)
	{
		multiply_mtx_MV( pP, result_1, result_2);
		pTmp = result_1;
		result_1 = result_2;
		result_2 = pTmp;
	}
	
	free(pValidStates);
	free_mtx_wrap(pP);
	free(dummy);
	free(result_2);
	
	return result_1;
}

/*****************************************************************************
name		: until
role       	: solve until formula.
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: bitset *psi: satisfaction relation for psi formula.
@param		: float subi: sub I.
@param		: float supi: sup I.
@return		: double *: result of the until formula for all states.
remark 	        : 
******************************************************************************/
double * until(bitset *phi, bitset *psi, float subi, float supi)
{
	const int mode = getRunMode();
	if( mode == CSL_MODE )
	{
		/*The CTMC case*/
		if(subi==0.0&&supi==0.0)
			return unbounded_until(phi, psi);
		else if(subi==0.0&&supi!=0.0)
			return bounded_until(phi, psi, supi);
		else if(subi!=0.0&&supi!=0.0)
			return interval_until(phi, psi, subi, supi);
	}
	else if( mode == PCTL_MODE )
	{
		/*The DTMC case*/
		if(subi==0.0&&supi==0.0)
			return dtmc_unbounded_until(phi, psi);
		else if(subi==0.0&&supi!=0.0)
			return dtmc_bounded_until(phi, psi, supi);
		else if(subi!=0.0&&supi!=0.0)
		{
			printf("This formula with general time bounds is not supported for PCTL\n");
			return (double *)calloc(phi->n, sizeof(double));
		}
	}
	else
	{
		printf("ERROR: Mode not supported for Until formula\n");
		exit(1);
	}
}

/*****************************************************************************
name		: dtmrm_bounded_until
role       	: solve the N-J-bounded until operator for DTMRM.
@param		: bitset *phi: SAT(phi).
@param		: bitset *psi: SAT(psi).
@param		: float supi: sup I, I is a time interval
@return		: double *: result of the N-J-bounded until operator for all states.
remark 	        : see (NOTE: slightly modified path_graph):
	1. S. Andova, H. Hermanns and J.-P. Katoen.
	Discrete-time rewards model-checked.
	FORMATS 2003, LNCS, Vol. 2791, Springer, pp. 88 - 103, 2003.
******************************************************************************/
static double * dtmrm_bounded_until(bitset *phi, bitset *psi, float subi, float supi, float subj, float supj)
{
	sparse *pM = get_state_space();
	//Translate phi and psi bitsets into array of node ids, where first elements defines the number of ids
	int *phi_set = count_set(phi), *psi_set = count_set(psi), i, j, k, l, size=phi->n, phi_size=phi_set[0], psi_size=psi_set[0];
	int state, col, count, found, num;
	double val, *reward=getRewards();
	path_graph *pg_n=NULL;       //a path graph of the last iteration
	path_graph *pg_n_plus=NULL;  //a new iteration path graph
	path_graph_ele *pge, *tpge;
	double *result=(double *)calloc(size, sizeof(double)), prob=0.0, value;
	bit get1b, get2b;
	
	//For every states, taken as initial state, compute probabilities
	for( i=0; i < size; i++ )
	{
		//First check whether we are already in a psi state if yes and
		//if 0 time and reward is allowed then according to the theory
		//we assign porob to 1 and skip the rest
		get_bit_val(psi, i, &get1b);
		//Although it looks ugly this is not an error, see the theory
		if( get1b.b &&
		    ( subi <= 0 ) && ( supi >= 0 ) &&
		    ( subj <= 0 ) && ( supj >= 0 ) )
		{
			result[i]=1.0;
			continue;
		}
		
		//Get an initial path graph, insert initial node for the current state and time 0
		pg_n = get_new_path_graph(size);
		insert_into_pg(pg_n, i, 0.0, 1.0);
		
		//Building the path graphs for all times in the time interval [subi,supi]
		for(j=0;j<supi;j++)
		{
			//Allocate path graph for time (j+1)
			pg_n_plus = get_new_path_graph(size);
			//For all phi states 
			for(k=1;k<=phi_size;k++)
			{
				state=phi_set[k];
				num = get_path_graph_num(pg_n, state);
				pge = get_path_graph_ele(pg_n, state);
				//Check if this phi state was reached at time j
				if(num>0)
				{
					//If YES then try to extend the corresponding path
					count=0;
					do
					{
						//Try non self loops first
						found=get_mtx_next_val(pM, state, &col, &val, &count);
						if(!found) break;
						//Check if the transition goes to a phi or a psi state
						get_bit_val(phi, col, &get1b);
						get_bit_val(psi, col, &get2b);
						if(get1b.b||get2b.b)
						{
							//Extend all paths ending in state at time j with this transition
							tpge=pge;
							for(l=0;l<num;l++,tpge++)
								insert_into_pg(pg_n_plus, col, reward[state]+tpge->reward,val*tpge->prob);
						}
					}while(found);
					found=get_mtx_diag_val(pM, state, &value);
					//If there is a self loop then it goes to phi or psi state :)
					if(found)
					{
						//Extend all paths ending in state at time j with this transition
						tpge=pge;
						for(l=0;l<num;l++,tpge++)
							insert_into_pg(pg_n_plus, state, reward[state]+tpge->reward,value*tpge->prob);
					}
				}
			}
			
			//If the time is right i.e. we are withing [subi,supi] then check for reached psi states!
			if( (j+1) >= subi && (j+1) <= supi )
			{
				//for all psi states
				for( k=1; k <= psi_size; k++ )
				{
					state=psi_set[k];
					num = get_path_graph_num(pg_n_plus, state);
					pge = get_path_graph_ele(pg_n_plus, state);
					//If a psi state was reached by some paths then
					//check all rewards with which we reached it...
					for( l=0; l < num;)
					{
						//If reward is within the required interval i.e. [subj,supj]
						if( ( pge->reward >= subj ) && ( pge->reward <= supj ) )
						{
							//accumulate the probability to get this reward and reach psi state
							prob+=pge->prob;
							//According to the Andova algorithms we do not extend
							//this path anymore, so we remove it from the path graph.
							num=delete_from_pg(pg_n_plus, state, l);
						}
						else {l++;pge++;}
					}
				}
			}
			free_path_graph(pg_n, size);
			pg_n = pg_n_plus;
			pg_n_plus=NULL;
		}
		result[i]=prob; prob=0.0;
		free_path_graph(pg_n, size);
	}
	return result;
}

/*****************************************************************************
name		: until_rewards
role       	: solve until formula with rewards.
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: bitset *psi: satisfaction relation for psi formula.
@param		: float subi: sub I (time).
@param		: float supi: sup I (time).
@param		: float subj: sub J (reward).
@param		: float supj: sup J (reward).
@return		: double *: result of the until formula for all states.
remark 	        : 
******************************************************************************/
double * until_rewards(bitset *phi, bitset *psi, float subi, float supi, float subj, float supj)
{
	const int mode = getRunMode();
	int i;
	double *ret_val;
	int mur;
	if( mode == PRCTL_MODE )
	{
		if(subi==0.0&&supi==0.0&&subj==0&&supj==0)
			return dtmc_unbounded_until(phi, psi);
		if(subi==0.0&&supi!=0.0&&subj==0&&supj==0)
			return dtmc_bounded_until(phi, psi, supi);
		if(supi!=0.0&&supj!=0)
			return dtmrm_bounded_until(phi, psi, subi, supi, subj, supj);
		else
		{
			printf("Something does seem to be wrong \n");
			return (double *)calloc(phi->n, sizeof(double));
		}
	}
	else if( mode == CSRL_MODE )
	{
		if(subi==0.0&&supi==0.0&&subj==0&&supj==0)
			return unbounded_until(phi, psi);
		if(subi==0.0&&supi!=0.0&&subj==0&&supj==0)
			return bounded_until(phi, psi, supi);
		if(subi!=0.0&&supi!=0.0&&subj==0&&supj==0)
			return interval_until(phi, psi, subi, supi);
		else if(subi==0.0&&subj==0.0&&supi!=0.0&&supj!=0.0)
		{
			ret_val = (double *)calloc(phi->n, sizeof(double));
			mur = get_method_until_rewards();
			if(mur==US)
				for(i=0;i<phi->n;i++)
					ret_val[i]=uniformization_sericola(i, phi, psi, supi, supj, 0);
			else if(mur==UQS)
				ret_val = uniformization_qureshi_sanders(phi, psi, supi, supj, 0);
			else if(mur==DTV)
				for(i=0;i<phi->n;i++)
					ret_val[i]=discretization(i, phi, psi, supi, supj, get_d_factor(), 0);
			return ret_val;
		}
		else
		{
			printf("not supported for CSRL.\n");
			return (double *)calloc(phi->n, sizeof(double));
		}
	}
	else
	{
		printf("ERROR: Mode not supported for Until formula\n");
		exit(1);
	}
}

/*****************************************************************************
name		: unbounded_next
role       	: solve the unbounded next operator.
@param		: bitset *phi: SAT(phi).
@return		: double *: result of the unbounded next operator for states.
remark 	        :
******************************************************************************/
static double * unbounded_next(bitset *phi)
{
	sparse *state_space = get_state_space();
	int size = state_space->rows, i;
	double * vec = (double *)calloc(size, sizeof(double));
	double * res = (double *)calloc(size, sizeof(double));
	bit getb;
	
	for(i=0;i<size;i++)
	{
		get_bit_val(phi,i,&getb);
		if(getb.b) vec[i] = 1.0;
	}
	set_sparse(state_space);
	multiplyMV(vec, res);	
	vec = get_e();
	for(i=0;i<size;i++) if(vec[i]) res[i]/=vec[i];
	free(vec);
	return res;
}

/*****************************************************************************
name		: bounded_next
role       	: solve the bounded next operator.
@param		: bitset *phi: SAT(phi).
@param		: float supi: sup I
@return		: double *: result of the bounded next operator for all states.
remark 	        :
******************************************************************************/
static double * bounded_next(bitset *phi, float supi)
{
	sparse *state_space = get_state_space();
	int size = state_space->rows, i;
	double * vec = (double *)calloc(size, sizeof(double)), vec_t;
	double * res = (double *)calloc(size, sizeof(double));
	bit getb;
	
	for(i=0;i<size;i++)
	{
		get_bit_val(phi,i,&getb);
		if(getb.b) vec[i] = 1.0;
	}
	set_sparse(state_space);
	multiplyMV(vec, res);	
	vec = get_e();
	for(i=0;i<size;i++)
		if(vec_t=vec[i])
	 		res[i]=(res[i]*(1-exp(-vec_t*supi)))/vec_t;
	free(vec);
	return res;
}

/*****************************************************************************
name		: interval_next
role       	: solve the interval next operator.
@param		: bitset *phi: SAT(phi).
@param		: float subi: sub I
@param		: float supi: sup I
@return		: double *: result of the interval next operator for all states.
remark 	        :
******************************************************************************/
static double * interval_next(bitset *phi, float subi, float supi)
{
	sparse *state_space = get_state_space();
	int size = state_space->rows, i;
	double * vec = (double *)calloc(size, sizeof(double)), vec_t;
	double * res = (double *)calloc(size, sizeof(double));
	bit getb;
	
	for(i=0;i<size;i++)
	{
		get_bit_val(phi,i,&getb);
		if(getb.b) vec[i] = 1.0;
	}
	set_sparse(state_space);
	multiplyMV(vec, res);
	vec = get_e();
	for(i=0;i<size;i++)
		if(vec_t=vec[i])
	 		res[i]=(res[i]*(exp(-vec_t*subi)-exp(-vec_t*supi)))/vec_t;
	free(vec);
	return res;
}

/*****************************************************************************
name		: interval_next_k
role       	: solve the interval next operator for CTMRMs (with K(s)).
@param		: bitset *phi: SAT(phi).
@param		: float subi: sub I
@param		: float supi: sup I
@return		: double *: result of the interval next operator for all states.
remark 	        : ********without impulse rewards at present***********
******************************************************************************/
static double * interval_next_k(bitset *phi, float *subk, float *supk)
{
	sparse *state_space = get_state_space();
	int size = state_space->rows, i;
	double * vec = (double *)calloc(size, sizeof(double)), vec_t;
	double * res = (double *)calloc(size, sizeof(double));
	bit getb;
	
	for(i=0;i<size;i++)
	{
		get_bit_val(phi,i,&getb);
		if(getb.b) vec[i] = 1.0;
	}
	set_sparse(state_space);
	multiplyMV(vec, res);
	vec = get_e();
	for(i=0;i<size;i++)
		if(vec_t=vec[i])
	 		res[i]=(res[i]*(exp(-vec_t*subk[i])-exp(-vec_t*supk[i])))/vec_t;
	free(vec);
	return res;
}

/**
* This method modelchecks the Xphi formula for the DTMC
* @param the phi states
* @return the probabilities
*/
static double * dtmc_unbounded_next(bitset *phi)
{
	sparse *state_space = get_state_space();
	const int size = state_space->rows;
	int i;
	double * result = (double *)calloc(size, sizeof(double));
	double * i_phi = (double *)calloc(size, sizeof(double));
	bit getb;
	
	for(i=0;i<size;i++)
	{
		get_bit_val(phi,i,&getb);
		if(getb.b) i_phi[i] = 1.0;
	}

	multiply_mtx_MV(state_space,i_phi, result);
	
	free(i_phi);
	
	return result;
}

/*****************************************************************************
name		: next
role       	: solve next formula.
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: float subi: sub I.
@param		: float supi: sup I.
@return		: double *: result of the next formula for all states.
remark 	        : 
******************************************************************************/
double * next(bitset *phi, float subi, float supi)
{
	const int mode = getRunMode();
	if( mode == CSL_MODE )
	{
		/*The CTMC case*/
		if(subi==0.0&&supi==0.0)
			return unbounded_next(phi);
		else if(subi==0.0&&supi!=0.0)
			return bounded_next(phi, supi);
		else if(subi!=0.0&&supi!=0.0)
			return interval_next(phi, subi, supi);
	}
	else if( mode == PCTL_MODE )
	{
		/*The DTMC case*/
		return dtmc_unbounded_next(phi);
	}
	else
	{
		printf("ERROR: Mode not supported for Next formula\n");
		exit(1);
	}
}

/*****************************************************************************
name		: next_rewards
role       	: solve next_rewards formula.
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: float subi: sub I.
@param		: float supi: sup I.
@param		: float subj: sub J.
@param		: float supj: sup J.
@return		: double *: result of the next formula for all states.
remark 	        : ********without impulse rewards at present***********
		  ********to be tested***********
******************************************************************************/
double * next_rewards(bitset *phi, float subi, float supi, float subj, float supj)
{
	const int mode = getRunMode();
	int i, size = (get_state_space())->rows;
	float *subk=NULL, *supk=NULL;
	double *reward=NULL;
	if( mode == CSRL_MODE )
	{
		/*The CTMC case*/
		if(subi==0.0&&supi==0.0&& subj==0 && supj==0)
			return unbounded_next(phi);
		else if(subi==0.0&&supi!=0.0&& subj==0 && supj==0)
			return bounded_next(phi, supi);
		else if(subi!=0.0&&supi!=0.0&& subj==0 && supj==0)
			return interval_next(phi, subi, supi);
		else if(supi!=0.0)
		{
			subk = (float *)calloc(size, sizeof(float));
			supk = (float *)calloc(size, sizeof(float));
			reward = getRewards();
			for(i=0;i<size;i++)
			{
				if(reward[i]==0){subk[i]=0.0;supk[i]=0.0;}
				else
				{
					subk[i]=subj/reward[i];
					supk[i]=supj/reward[i];
					if(subk[i]>supi||supk[i]<subi)
					{subk[i]=0;supk[i]=0;}
					else if(subk[i]<=subi&&supk[i]>=supi)
					{subk[i]=subi;supk[i]=supi;}
					else if(subk[i]>=subi&&subk[i]<=supi&&supk[i]>=supi)
						supk[i]=supi;
					else if(subk[i]<=subi&&supk[i]<=supi&&supk[i]>=subi)
						subk[i]=subi;
					else if(subk[i]>=subi&&subk[i]<=supi&&supk[i]<=supi&&supk[i]>=subi);
				}
			}			
			return interval_next_k(phi, subk, supk);
		}
		else
		{
			printf("Formula is not supported\n");
			return (double *)calloc(size, sizeof(double));
		}
	}
	else if( mode == PRCTL_MODE)
	{
		/*Only unbounded next is allowed (see parser for
		restriction of other cases) - The DTMC case*/
		if(subi==0 && supi==0 && subj==0 && supj==0)
			return dtmc_unbounded_next(phi);
		else
		{
			printf("Formula is not supported in PRCTL mode\n");
			return (double *)calloc(size, sizeof(double));
		}
	}
	else
	{
		printf("ERROR: Mode not supported for Next formula\n");
		exit(1);
	}
}
