/**
*	WARNING: Do Not Remove This Section
*
*	$RCSfile: uniformization_qureshi_sanders.c,v $
*	$Revision: 1.23 $
*	$Date: 2006/02/13 22:42:04 $
*	$Author: zapreevis $
*	
*	MRMC is a model checker for discrete-time and continuous-time Markov reward models.
*	It supports reward extensions of PCTL and CSL (PRCTL and CSRL), and allows for the
*	automated verification of properties concerning long-run and instantaneous rewards
*	as well as cumulative rewards.
*	
*	Copyright (C) The University of Twente, 2004-2006.
*	Authors: Maneesh Khattri, Ivan Zapreev
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*	Our contacts:
*		Formal Methods and Tools Group, University of Twente,
*		P.O. Box 217, 7500 AE Enschede, The Netherlands,
*		Phone: +31 53 4893767, Fax: +31 53 4893247,
*		E-mail: mrmc@cs.utwente.nl
*
*	Source description: Uniformization based algorithm by Qureshi&Sanders for CSRL - U.
*	Uses: DEF: bitset.h, sparse.h, label.h, runtime.h
*		LIB: bitset.c, sparse.c, label.c, runtime.c
*	Remarks: Uniformization based algorithm by Qureshi&Sanders for CSRL - U.
*
*	For discussion see:
*		1. M.A. Qureshi and W.H.Sanders. Reward Model Solution Methods with Impulse and Rate Rewards:
*		An Algorithm and Numerical Results. Performance Evaluation, 20(4), 1994.
*		2. M.A. Qureshi and W.H. Sanders. A new methodology for calculating distributions of reward
*		accumulated during a finite interval. Fault-Tolerant Computing Symposium, IEEE CS Press,
*		pp. 116-125, 1996.
*
*	FOR OMEGA SEE
*		3. M.C. Diniz, E. de Souza e Silva and H.R. Gail. Calculating the distribution of a linear
*		combination of uniform of order statistics. INFORMS Journal on Computing: 14(2), pp. 124-131, 2002.
*
*	FOR Model Checking Markov Reward Models with Impulse Rewards see:
*		4. M. Khattri, R. Pulungan, Model Checking Markov Reward Models with Impulse Rewards, M.Sc. Thesis,
*		UT-FMG-2004.
*		5. L. Cloth, J.-P. Katoen, M. Khattri, R. Pulungan, Model Checking Markov Reward Models with Impulse Rewards,
*		DSN-PDS-05.
*/

# include "macro.h"
#ifdef MAC_OS
	# include <stdlib.h>
#else
	# include <malloc.h>
#endif
#include "bitset.h"
#include "sparse.h"
#include "label.h"
#include "runtime.h"
#include <math.h>
#include "kjstorage.h"
#include "transient.h"

  /* GLOBAL */
  double maxPoi;
  int *index_;
  double *omIteration;
  double *Poisson;
  double *pathErr;
  /* GLOBAL */

/*****************************************************************************
name		: get_dsr
role       	: get the list of distinct state rewards from rew_matrix.(Descending order)
@param		: double *rew_matrix: vector containing the original state rewards.
@param		: bitset *valid_states: transient states after making states absorbing.
@return		: double *dsr: vector of distinct state rewards.
remark 	        : the output is a sorted list, so a translation of index(index_) is required.
******************************************************************************/
double *get_dsr(double *rew_matrix, bitset *valid_states, int rows, int *ndsr_)
{
	int i, j;
	double *dsr = (double *)calloc(rows, sizeof(double));
	double rew;
	int ndsr=1, l, u, m;
	bit get1b;
	int found=0;
	for(i=0;i<rows;i++)
	{
		get_bit_val(valid_states, i, &get1b);
		if(get1b.b)
		{
			rew=rew_matrix[i];
			l=0;u=ndsr-1;found=0;m=0;
			while(l<=u)
			{
				m=(l+u)/2.0;
				if(rew==dsr[m]){found=1; index_[i]=m; break;}
				if(rew>dsr[m]) u=m-1;
				if(rew<dsr[m]) l=m+1;
			}
			if(found==0)
			{
				if((m==(ndsr-1)&&(rew>dsr[m]))||m<(ndsr-1))
				{
					/*shift*/
					ndsr++;
					for(j=(ndsr-1);j>m;j--)
						dsr[j]=dsr[j-1];
					for(j=0;j<rows;j++)
						if(index_[j]>=m)
							index_[j]++;
					/*put the new one*/
					dsr[m]=rew;
					index_[i]=m;
				}
				else
				{
					dsr[ndsr++]=rew;
					index_[i]=ndsr;
				}
			}
		}
	}
	dsr = (double *)realloc(dsr, ndsr*sizeof(double));
	*ndsr_=ndsr;
	/* FOR TEST ONLY
	printf("DSR = {");
	for(i=0;i<ndsr;i++)
		printf("%e,",dsr[i]);
	printf("}\n");
	printf("Index = {");
	for(i=0;i<rows;i++)
		printf("%d,",index_[i]);
	printf("}\n");*/
	return dsr;
}

/* pivot: all indices less than pivot are in G, 
pivot and greater indices are in L. */
void omIt(int ndsr, int *tmp_k, double r_dash, int j, int pivot, int cols, double *dsr)
{
	double prevValue = 1.0, val_i, val_j;
	int *tempG = (int *)calloc(ndsr, sizeof(int)), count, i;
	
	for(count=0; count<cols;count++)
	for(i=0;i<pivot;i++)
	{
		if(tempG[i]<tmp_k[i])
		{
			tempG[i]++;
			val_i = dsr[i];
			val_j = dsr[j];
			omIteration[count] = ((val_i-r_dash)/(val_i-val_j))*omIteration[count]+((r_dash-val_j)/(val_i-val_j))*prevValue;
			prevValue = omIteration[count];
			break;
		}
	}
	free(tempG);tempG=NULL;
}

double omega(int *tmp_k, double r_dash, int pivot, int ndsr, double *dsr)
{
	int cols = 0, rows=0, count, *tempL, j;
	for(count=pivot;count<ndsr;count++)
		rows+=tmp_k[count];
	for(count=0;count<pivot;count++)
		cols+=tmp_k[count];
	if(rows!=0 && cols==0)
		return 1.0;
	else if (rows==0 && cols!=0)
		return 0.0;

	tempL =(int *)calloc(ndsr, sizeof(int));
	omIteration = (double *)calloc(cols, sizeof(double));
	for(count=0; count<rows;count++)
	{
		for(j=pivot;j<ndsr;j++)
		{
			if(tempL[j]<tmp_k[j])
			{
				tempL[j]++;
				omIt(ndsr, tmp_k, r_dash, j, pivot, cols, dsr);
				break;
			}
		}
	}
	free(tempL); tempL=NULL;
	return omIteration[cols-1];
}

/*****************************************************************************
  Method Name       : findPivot
  Parameters        : double r_dash, double * dsr, int ndsr
  Return Type       : pivot: int
  Remarks           : 	all indices less than pivot are in G
  			pivot and all indices greater than pivot are in L
  			Ref: Solution for Order Statistics by Diniz, de Souza e Silva & H.R. Gail
*****************************************************************************/
  /* Since di = (a(i)+a(i+1)...+a(n))
  In our problem a(i) = r(i) - r(i+1)
  therefore
        di = ( (r(i)-r(i+1)) + (r(i+1)-r(i+2)) + ...+ (r(n)-r(n+1)) )
        and since r(n+1) = 0; (Provided there is always absorbing state)
  hence
  	di = r(i)
  since all r(i)'s are distinct
  	ci = di = ri
  
  all indices less than pivot are in G
  pivot and all indices greater than pivot are in L  */
 int findPivot(double r_dash, double *dsr, int ndsr)
 {
 	int i=0;
 	 for(;i<ndsr;i++)
 	 	if(dsr[i]<=r_dash)
 	 		break;
 	return i;
 }

/*****************************************************************************
  Method Name       : maxPoisson
  Parameters        : int n
  Return Type       :
  Remarks           : calculate poisson probability.
*****************************************************************************/
void maxPoisson(int n, double lambda, double mTime, int *psize)
{
	int i;
	if(*psize==0)
	{
		maxPoi = exp(-(lambda*mTime));
		(*psize)++;
		Poisson[0]=maxPoi;
		pathErr[0]=maxPoi;
		/*printf("psize = %d, maxPoi=%e\n", *psize, maxPoi);*/
	}
	if(*psize<=n)
	{
		Poisson = (double *)realloc(Poisson, (n+1)*sizeof(double));
		pathErr = (double *)realloc(pathErr, (n+1)*sizeof(double));
		for(i=*psize;i<=n;i++)
		{
			maxPoi *= (lambda*mTime)/(i);
			(*psize)++;
			Poisson[i] = maxPoi;
			pathErr[i] = pathErr[i-1]+maxPoi;
			/*printf("Poisson[%d]=%e, psize = %d, maxPoi=%e\n", i, Poisson[i], *psize, maxPoi);*/
		}
	}
}

/*****************************************************************************
  Method Name       : errorBounds
  Parameters        : int n, double prob
  Return Type       :
  Remarks           : calculate error bounds cumulatively given one discarded path at a time.
*****************************************************************************/
void errorBounds(int n, double prob, double *tot_err, double lambda, double mTime, int *psize)
{
	maxPoisson(n, lambda, mTime, psize);
	if(n>0)
		*tot_err += prob * (1-pathErr[n-1]);
	else
		*tot_err += prob;
}

void testMaxPoisson(double lambda, double supi, int *psize)
{
	int n=100, i;
	maxPoisson(n, lambda, supi, psize);
	for(i=0;i<n;i++)
		printf("Poisson[%d] = %e, pathErr = %e\n", i, Poisson[i], pathErr[i]);
}

/*****************************************************************************
  Method Name       : dfpg
  Parameters        : w - Probability Threshold
  Return Type       :  The algorithm computes the probability of 
                       all paths in the uniformized DTMC (D). If we multiply the probability of 
                       some path (in D) with the Poisson probability for a path of length 'n', 
                       given that n is the length of the path in question, then we obtain the 
                       probability of the path to be in the CTMC from which you get (D). But these 
                       path probabilities are generated one-by-one given (D) which can be of any 
                       length. So, it is necessary to stop the computation at some point. This is 
                       where "w" - probability threshold - comes in, only paths whose probability 
                       is more than 'w' are considered in the computation.
  Remarks           : depth first path generation
*****************************************************************************/
void dfpg(int * global_k, sparse *abs, sparse *rewi_matrix, int n, int state, double p, double r_impulse, double w, double lambda, double supi, double supj, int *psize, double *tot_err, kjstruc *kjstruct, bitset *psi)
{
	int found=0;
	maxPoisson(n, lambda, supi, psize);
	double pr = p*Poisson[n];
	int rows = psi->n;
	bit b;
	/*printf("pr=%e\n",pr);*/
	if(supj<r_impulse || pr<w)
	{
		errorBounds(n, pr, tot_err, lambda, supi, psize);
		return;
	}
	get_bit_val(psi, state, &b);
	if(b.b)
	{
		add_new_kjnode(kjstruct, global_k, r_impulse, p);
		return;
	}
	int count=0, col=0.0;
	double val=0.0, value;
	while(get_mtx_next_val(abs, state, &col, &val, &count)!=0)
	{
		if(val<=0) continue;
		int kIndex = index_[col];
		global_k[kIndex]++;
		value=0.0;
		if(rewi_matrix!=NULL)
			get_mtx_val(rewi_matrix, state, col, &value);
		dfpg(global_k, abs, rewi_matrix, n+1, col, p * val, r_impulse+value, w, lambda, supi, supj, psize, tot_err, kjstruct, psi);
		global_k[kIndex]--;
	}
	found=get_mtx_diag_val(abs, state, &val);
	if(found!=0)
	{
		int kIndex = index_[col];
		global_k[kIndex]++;
		value=0.0;
		dfpg(global_k, abs, rewi_matrix, n+1, state, p * val, value, w, lambda, supi, supj, psize, tot_err, kjstruct, psi);
		global_k[kIndex]--;
	}
}


/*****************************************************************************
name		: uniformization_qureshi_sanders
role       	: solve t-r-bounded until formula in MRM without impulse rewards
		  using uniformization by qureshi & sanders
@param		: int state: the starting state.
@param		: bitset *phi: satisfaction relation for phi formula.
@param		: bitset *psi: satisfaction relation for psi formula.
@param		: double supi: sup I.
@param		: double supj: sup J.
@param		: int: print.
@return		: double: result of the until formula for one state.
remark 	        : dfpg
******************************************************************************/
double * uniformization_qureshi_sanders(bitset *phi, bitset *psi, double supi, double supj, int print)
{
	sparse *r_matrix = get_state_space();
	sparse *rewi_matrix = get_impulse_rewards();
	double *e = get_e();
	double *rew_matrix = getRewards();
	double w=get_w(), tot_err=0.0;
	int rows = phi->n, idx1, idx2, ki,jj;
	bit get1b, get2b;
	bitset *not_psi=not(psi);
	bitset *valid_states = and(phi, not_psi);
	int valid_number = count_non_zero(valid_states), ndsr;
	index_=(int *)calloc(rows, sizeof(int));
	double *dsr = get_dsr(rew_matrix, valid_states, rows, &ndsr);
	double * diag = (double *)calloc(rows, sizeof(double));
	sparse *state_space = get_state_space();
	double lambda=0, total_prob=0.0;
	int non_absorbing=0, psize=0;
	kjstruc *kjstruct=NULL;
	sparse *abs = ab_state_space(state_space, valid_states, &lambda, diag, &non_absorbing);
	double *result=(double *)calloc(phi->n, sizeof(double));
	int * global_k=(int *)calloc(ndsr, sizeof(int));
	Poisson=(double*)calloc(1, sizeof(double));
	pathErr=(double*)calloc(1, sizeof(double));

		/* R-E(s) */
		set_sparse(abs);
		sub_diagonal(diag);
		/* uniformize : lambda>0*/
		if ( lambda == 0 )
		{
			printf("ERROR: Uniformization rate is 0 in Qureshi Sanders\n");
			exit(1);
		}
		mult_const(1/lambda);
		add_cons_diagonal(1.0);

	/*print_mtx_sparse(abs);
	printf("lambda = %e, w = %e\n", lambda, w);*/
	for(ki=0;ki<phi->n;ki++)
	{
		total_prob=0.0; tot_err=0.0;
		kjstruct = get_new_kjstruc(ndsr);
		kjnode *trav;
		int kIndex = index_[ki];
		for(jj=0;jj<ndsr;jj++)
			global_k[jj] = 0;
		global_k[kIndex]++;
		dfpg(global_k, abs, rewi_matrix, 0.0, ki, 1.0, 0.0, w, lambda, supi, supj, &psize, &tot_err, kjstruct, psi);
		global_k[kIndex]--;     //This doed not do a thing !!!!!
		//testMaxPoisson(lambda, supi, &psize);
		//print_kjstorage(kjstruct);
		trav = kjstruct->kjnodes;
		while(trav!=NULL)
		{
			double r_dash = supj/supi-trav->acir/supi;
			int pivot = findPivot(r_dash, dsr, ndsr);
			int i, n=0;
			int * tmp_k=trav->k;
			for(i=0; i<ndsr; i++)
				n+=tmp_k[i];
			--n;
			maxPoisson(n, lambda, supi, &psize);
			double pr = trav->prob *  Poisson[n];
			double om = omega(tmp_k, r_dash, pivot, ndsr, dsr);
			int cols = 0;
			for(i=0; i<pivot; i++)
				cols+=tmp_k[i];
			while(pr>=w && r_dash>=0.0)
			{
				total_prob += pr * (om);
				tmp_k[ndsr-1]++;
				pr = pr * ((lambda*supi)/++n);
				if(pr>=w && cols>0)
				{
					omIt(ndsr, tmp_k, r_dash, ndsr-1, pivot, cols, dsr);
					om = omIteration[cols-1];
				}
			}
			errorBounds(n, trav->prob, &tot_err, lambda, supi, &psize);
			trav=trav->next;
			free(omIteration); omIteration=NULL;
		}
		free_kjstorage(kjstruct);
		kjstruct=NULL;
		//NOTE: indexes of states from the user point of view start from 1
		printf("State = %d, Probabilty = %e, Error Bound = %e \n", ki+1, total_prob, tot_err);
		if(tot_err>=0.1) printf("WARNING: The Probability Threshold w = %e is to big!\n",tot_err);
		result[ki] = total_prob;
	}
	
	/* NOTE: operations on diagonals are not required */
	set_sparse(abs);
	mult_const(lambda);
     
	free_abs(abs); /*free allocated memory*/

	free(not_psi);
	free(valid_states);
	free(Poisson);Poisson=NULL;
	free(pathErr);pathErr=NULL;
	free(global_k); global_k=NULL;
	free(index_);index_=NULL;
	
	return result;
}
